/*
Breaboard simulator
Copyright (C) 2020  Balázs Dura-Kovács

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#include "spice.h"
#include <QDebug>
#include <QString>
#include <QStringList>
#include <QMap>
#include <QEventLoop>
#include <QTimer>
#include <math.h>
#include <algorithm>

#include "sharedspice.h"



// use .option noopac
// https://sourceforge.net/p/ngspice/discussion/ngspice-tips/thread/9605d87518/

Spice::Spice(){
    running = false;
    ngSpice_Init(&recieve_char, &recieve_stat, &ngexit,
				 &recieve_data, &recieve_init_data, &ngrunning, (void*)this);
    timer2.setSingleShot(true);
    connect(&timer, &QTimer::timeout, this, &Spice::doRestart);
    connect(&timer2, &QTimer::timeout, this, &Spice::doRestart);
}

// measurements: list of wanted values for CH1A, CH1B, CH2A, CH2B, EXTA, EXTB, VOLTMETERA, VOLTMETERB respectively
// (measured value: CH1A-CH1B)
// V(0) means ground
// eg. {"V(2)", "V(0)", "V(1)", "V(0)","V(3)", "V(0)"}
void Spice::start(QString circuit, qreal timestep, qreal frequency, QStringList wanted_measurements){
    this->wanted_measurements = wanted_measurements;
    this->circuit = circuit;
    this->timestep = timestep;
    this->nextFrequency = frequency;

    if(running){
        restartOnPause = true;
        timer.start(1000);

        return;
    }
    restartOnPause = false;


    restart();
}

void Spice::restart(){
    running = true;
    paused = false;

    ngSpice_Command(NULL);

    frequency = nextFrequency;
    const qreal period = 1/this->frequency;

    // limited by space
    const qreal maxtime = MAX_FRAMES * SAMPLES_PER_FRAME * this->timestep;
    // min number of frames + finish period
    const qreal idealtime = MIN_FRAMES * SAMPLES_PER_FRAME * this->timestep + 1.1 * period;
    const qreal simulation_length = std::min(maxtime, idealtime);
    // avoid transient behaviour
    const qreal simulation_start = 3*this->timestep*SAMPLES_PER_FRAME;
    // if end of period not found, stop just before the end of the simulation stops
    stop_time = maxtime + simulation_start - 10 * this->timestep;


    //replace parameters in spice code
    circuit = this->circuit.replace("__TIMESTEP__",QString::number(this->timestep,'f',15));
    circuit = circuit.replace("__TSTART__",QString::number(simulation_start,'f',15));
    circuit = circuit.replace("__TEND__",QString::number(simulation_start+simulation_length,'f',15));
    end_time = simulation_start+simulation_length - 10 * this->timestep;

    //qDebug() << circuit;

    // convert circuit description to array of lines
    QStringList lines = circuit.split('\n');
    char** circarray = (char**)malloc(sizeof(char*) * (lines.length()+1));
    for(int i=0; i<lines.length(); i++){
        const std::string& stdS = lines[i].toStdString();
        circarray[i] = strdup(stdS.c_str());
    }
    circarray[lines.length()] = NULL;
    ngSpice_Circ(circarray);


    firstCycle = true;

    CH1A_index = -1;
    CH1B_index = -1;
    CH2A_index = -1;
    CH2B_index = -1;
    EXTA_index = -1;
    EXTB_index = -1;
    TIME_index = -1;
    VOLTMETERA_index = -1;
    VOLTMETERB_index = -1;
    start_time = 0;
    rmsVoltage2 = 0;
    rmsSum = 0;

    buffer_size = 0;
    buffer_ready = false;
    seek_phase = false;



    //qDebug() << "spice started";
    #ifdef TARGET_OS_MAC
        ngSpice_Command(strdup("run"));
    #else
        ngSpice_Command(strdup("bg_run"));
    #endif
}

void Spice::doRestart(){
    if(running){
        return;
    }

    timer.stop();
    timer2.stop();
    //qDebug() << "tick2";

    if(restartOnPause){
        restartOnPause = false;
        timer2.start(500);
        return;
    }


    //qDebug() << "restart";

    restart();
}




void Spice::pause(){
    if(running){
        running = false;
        paused = true;
    }
}

void Spice::resume(){
    if(!running && paused){
        running = true;
        paused = false;
        //qDebug() << "spice resumed";
        #ifdef TARGET_OS_MAC
            ngSpice_Command(strdup("resume"));
        #else
            ngSpice_Command(strdup("bg_resume"));
        #endif
    }
}


// total_count: index of data point as counted by ngspice
void Spice::recieveData(vecvaluesall* data){
    if(buffer_ready){
        if(data->vecsa[TIME_index]->creal >= end_time){
            this->pause();
        }
        return;
    }
    if(firstCycle){
        // map recieved values to wanted measurements
        firstCycle = false;

        QMap<QString,size_t> nameToIndex;
        int* indices[8] = {&CH1A_index,&CH1B_index,&CH2A_index,&CH2B_index,
                           &EXTA_index,&EXTB_index,&VOLTMETERA_index,&VOLTMETERB_index};

        for(int i=0; i<data->veccount; i++){
            nameToIndex[data->vecsa[i]->name] = i;
        }

        for(int i=0; i<8; i++){
            if(nameToIndex.contains(wanted_measurements[i])){
                *(indices[i]) = nameToIndex[wanted_measurements[i]];
            }
        }

        if(nameToIndex.contains("time")){
            TIME_index = nameToIndex["time"];
            start_time = data->vecsa[TIME_index]->creal;
        }

    }



    CH1[buffer_size]  = (CH1A_index  >= 0 ? data->vecsa[CH1A_index]->creal  : 0) - (CH1B_index  >= 0 ? data->vecsa[CH1B_index]->creal  : 0);
    CH2[buffer_size]  = (CH2A_index  >= 0 ? data->vecsa[CH2A_index]->creal  : 0) - (CH2B_index  >= 0 ? data->vecsa[CH2B_index]->creal  : 0);
    EXT[buffer_size]  = (EXTA_index  >= 0 ? data->vecsa[EXTA_index]->creal  : 0) - (EXTB_index  >= 0 ? data->vecsa[EXTB_index]->creal  : 0);
    buffer_size++;

    //compute RMS voltage
    if(voltMeter){
        qreal value = (VOLTMETERA_index  >= 0 ? data->vecsa[VOLTMETERA_index]->creal  : 0) - (VOLTMETERB_index  >= 0 ? data->vecsa[VOLTMETERB_index]->creal  : 0);
        value = value*value;
        /*qreal a = 1/buffer_size;
        rmsVoltage2 = a * rmsVoltage2 + (1 - a) * value;*/
        rmsSum += value;
        rmsVoltage2 = rmsSum / buffer_size;
    }

    if(buffer_size % 100 == 0){
        //qDebug() << buffer_size;
    }

    if(data->vecsa[TIME_index]->creal >= stop_time){
        buffer_ready = true;
        emit bufferLoaded();
        this->pause();
    }else if(!seek_phase && buffer_size >= MIN_FRAMES * SAMPLES_PER_FRAME){
        seek_phase = true;
        last_phase = -1; // smaller than any real phase
    }else if(seek_phase){ // find minimum in phase, that's where the pattern will repeat
        qreal time = data->vecsa[TIME_index]->creal;
        qreal intpart;
        qreal phase = modf((time-start_time)*frequency,&intpart); // this is sawtooth, only need to find where next value is smaller than previous one

        if(phase < last_phase){
            buffer_ready = true;
            emit bufferLoaded();
        }

        last_phase = phase;

    }


}


int Spice::recieve_char(char * str, int id, void * that){
    //qDebug() << "recieved " << str;
	return 0;
}

int Spice::recieve_stat(char* status, int id, void* that){
    ((Spice *)that)->appViewer->setTitle(QString("Oscilloscope: ") + QString(status));
    Spice * obj = (Spice*) that;
    //qDebug() << "status " << status;
	return 0;
}

int Spice::ngexit(int status, bool unload, bool exit, int id, void* that){
    //qDebug() << "exit: " << status;
	return 0;
}

int Spice::recieve_data(vecvaluesall* data, int numstructs, int id, void* that){
    ((Spice *)that)->recieveData(data);
    //printf("data recieved: %f\n", data->vecsa[0]->creal);
    /*printf("data recieved. count: %d, ", data->veccount);
    for(int i=0; i<data->veccount; i++){
        printf("%s: %f, ", data->vecsa[i]->name, data->vecsa[i]->creal);
    }
    printf("\n");*/
	return 0;
}

int Spice::recieve_init_data(vecinfoall* data, int id, void* that){
    //qDebug() << "init data recieved from: " << id;
	return 0;
}

void Spice::emitSpiceHalted(){
    emit spiceHalted();
}

void Spice::emitBufferLoaded(){
    emit bufferLoaded();
}

int Spice::ngrunning(bool running, int id, void* that){
    if(running){
        //("ng is running\n");
	}else{
        //qDebug("ng is not running\n");
        emit ((Spice *)that)->spiceHalted();
    }
	return 0;
}
