/*
Breaboard simulator
Copyright (C) 2020  Balázs Dura-Kovács

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#include <QtQuick/QQuickView>

#include <chrono>
#include <random>
#include "signalgeneratorproperties.h"
#include "siphash.h"





SignalGeneratorProperties::SignalGeneratorProperties(QQuickView *appViewer, QObject *parent) :
	QObject(parent),
    m_viewer(appViewer)
{
    QObject::connect(m_viewer, &QQuickView::statusChanged, this, &SignalGeneratorProperties::qmlStatusChanged);
}

void SignalGeneratorProperties::qmlStatusChanged(QQuickView::Status status){
    if(status == QQuickView::Status::Ready){
        this->init();
    }
}

void SignalGeneratorProperties::init(){
    sgView = m_viewer->findChild<QObject*>("signalGenerator");
    this->update();

}

void SignalGeneratorProperties::update(){
    frequency = sgView->property("frequency").toDouble();
    amplitude = sgView->property("amplitude").toDouble();
    dc_offset = sgView->property("dcOffset").toDouble();
    waveform = sgView->property("waveform").toString();

    emit changed();
}

QString SignalGeneratorProperties::formatFrequency(qreal freq){
    char buffer[15];
    sprintf(buffer, "%09.1f", freq);
    return QString(buffer).replace(',', '.');
}

// generate spice voltage source
QString SignalGeneratorProperties::toSpice(QString student_number){

    // randomise frequency deterministically
    const std::string& toHashStd = student_number.toStdString();

    qint64 hash = 0;
    uint8_t hash_s[8];
    const uint8_t k[16] = {3, 1, 4, 1, 5, 9, 2, 6, 5, 3, 5, 8, 9, 7, 9, 3};
    siphash((uint8_t * )toHashStd.c_str(), toHashStd.length(), k, (uint8_t *) hash_s, 8);
    memcpy(&hash, hash_s, 8);
    generator.seed(hash);

    //+/- 20ppm
    std::normal_distribution<qreal> gauss_distribution = std::normal_distribution<qreal>(1.,20./1e6);
    realFrequency = frequency * gauss_distribution(generator);
    qreal f = realFrequency;

    if(waveform == "SIN"){
        return "SIN(" + QString::number(dc_offset,'f',5) + " " + QString::number(amplitude,'f',5) + " " + QString::number(f,'f',15) + " 0 0)";
    }

    qreal period = 1.0 / f;
    if(waveform == "SQUARE"){
        return "PULSE(" +
                QString::number(dc_offset-amplitude,'f',5) + " " + //V1
                QString::number(dc_offset+amplitude,'f',5) + " " + //V2
                "0.0 " + //TD
                "2NS " + //TR
                "2NS " + //TF
                QString::number(period/2 - 2e-9,'f',15) + " " + //PW
                QString::number(period,'f',15) + " " + //PER
                "0.0)"; //PHASE
    }

    if(waveform == "SAWTOOTH"){
        return "PWL(0 " + //T1
                QString::number(dc_offset-amplitude,'f',5) + " " + //V1
                QString::number(period,'f',15) + " " + //T2
                QString::number(dc_offset+amplitude,'f',5) + ") r=0"; //V2
    }

    if(waveform == "TRIANGLE"){
        return "PWL("
                "0 " + //T1
                QString::number(dc_offset-amplitude,'f',5) + " " + //V1
                QString::number(period/2,'f',15) + " " + //T2
                QString::number(dc_offset+amplitude,'f',5) + " " + //V2
                QString::number(period,'f',15) + " " + //T3
                QString::number(dc_offset-amplitude,'f',5) + ") r=0"; //V1
    }

    return "";

}
