/*
Breaboard simulator
Copyright (C) 2020  Balázs Dura-Kovács

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#include <QtQuick/QQuickView>

#include <chrono>
#include <random>
#include <cmath>
#include "multimeterproperties.h"
#include "siphash.h"





MultimeterProperties::MultimeterProperties(QQuickView *appViewer, QObject *parent) :
	QObject(parent),
    m_viewer(appViewer)
{
    QObject::connect(m_viewer, &QQuickView::statusChanged, this, &MultimeterProperties::qmlStatusChanged);
}

void MultimeterProperties::qmlStatusChanged(QQuickView::Status status){
    if(status == QQuickView::Status::Ready){
        this->init();
    }
}

void MultimeterProperties::init(){
    mmView = m_viewer->findChild<QObject*>("multimeter");
    this->update();

}

void MultimeterProperties::update(){
    QString ms = mmView->property("mode").toString();
    
    if(ms == "AC true RMS 600.0V"){
        mode = MultimeterProperties::modes::AC_600V;
    }else if(ms == "AC true RMS 60.00V"){
        mode = MultimeterProperties::modes::AC_60V;
    }else if(ms == "AC true RMS 6.000V"){
        mode = MultimeterProperties::modes::AC_6V;
    }else if(ms == "AC true RMS 600.0mV"){
        mode = MultimeterProperties::modes::AC_600mV;
    }else{
        mode = MultimeterProperties::modes::OFF;
    }

    cachedValue = 0;
    emit changed();
}



// change display value
void MultimeterProperties::setValue(qreal value, QString student_number){
    if(cachedValue == value){
        return;
    }
    cachedValue = value;
    if(mode == OFF || signalGeneratorProperties->frequency > 1000 || signalGeneratorProperties->frequency < 45){
        mmView->setProperty("value", "0.0");
        return;
    }

    value = sqrt(value);
    //qDebug() << value;

    if(value > 800){
        mmView->setProperty("value", "0.0");
        return;
    }

    if(value > 80 && (mode == AC_60V || mode == AC_6V || mode == AC_600mV)){
        mmView->setProperty("value", "0.0");
        return;
    }

    if(value > 8 && ( mode == AC_6V || mode == AC_600mV)){
        mmView->setProperty("value", "0.0");
        return;
    }

    if(value > 0.8 && mode == AC_600mV){
        mmView->setProperty("value", "0.0");
        return;
    }

    // randomise systematic error deterministically
	const std::string& toHashStd = student_number.toStdString();

    qint64 hash = 0;
    uint8_t hash_s[8];
    const uint8_t k[16] = {6, 1, 4, 1, 5, 9, 2, 6, 5, 3, 5, 8, 9, 7, 9, 3};
    siphash((uint8_t * )toHashStd.c_str(), toHashStd.length(), k, (uint8_t *) hash_s, 8);
    memcpy(&hash, hash_s, 8);
    generator.seed(hash);

    qreal accuracy = 0.01;
    if(signalGeneratorProperties->frequency > 500){
        accuracy = 0.02;
    }

    std::normal_distribution<qreal> gauss_distribution = std::normal_distribution<qreal>(0.,1.0);
    value *= (1 + accuracy * gauss_distribution(generator));

    int digits = 0; // to be displayed after decimal point
    switch(mode){
        case AC_600V:
            accuracy = 0.3;
            digits = 1;
            break;
        case AC_60V:
            accuracy = 0.03;
            digits = 2;
            break;
        case AC_6V:
            accuracy = 0.003;
            digits = 3;
            break;
        case AC_600mV:
            accuracy = 0.3;
            digits = 1;
            value *= 1000;
            break;
        default:
            accuracy = 0;
    }

    value += gauss_distribution(generator) * accuracy;

    QString dv = QString::number(value,'f',digits);
    mmView->setProperty("value", dv);

}

