/*
Breaboard simulator
Copyright (C) 2020  Balázs Dura-Kovács

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#include <QtWidgets/QApplication>
#include <QtQml/QQmlContext>
#include <QtQuick/QQuickView>
#include <QtQml/QQmlEngine>
#include <QtQml/QQmlApplicationEngine>
#include <QtCore/QDir>

#include "datasource.h"
#include "scopeproperties.h"
#include "signalgeneratorproperties.h"
#include "breadboardproperties.h"
#include "multimeterproperties.h"

#define xstr(s) str(s)
#define str(s) #s

int main(int argc, char *argv[])
{

	//QCoreApplication::setAttribute(Qt::AA_EnableHighDpiScaling);

	// Qt Charts uses Qt Graphics View Framework for drawing, therefore QApplication must be used.
    QApplication app(argc, argv);

    QQuickView viewer;
    viewer.setTitle(QStringLiteral("Oscilloscope"));

    QQuickView sgViewer;
    sgViewer.setTitle(QStringLiteral("Signal generator"));

    QQuickView breadboardViewer;
    breadboardViewer.setTitle(QStringLiteral("Breadboard"));
    breadboardViewer.setWidth(930);
    breadboardViewer.setHeight(750);

    QQuickView multimeterViewer;
    multimeterViewer.setTitle(QStringLiteral("Multimeter"));


    QObject::connect(viewer.engine(), &QQmlEngine::quit, &viewer, &QWindow::close);
    QObject::connect(sgViewer.engine(), &QQmlEngine::quit, &sgViewer, &QWindow::close);
    QObject::connect(breadboardViewer.engine(), &QQmlEngine::quit, &breadboardViewer, &QWindow::close);
    QObject::connect(multimeterViewer.engine(), &QQmlEngine::quit, &multimeterViewer, &QWindow::close);

	ScopeProperties scopeProperties(&viewer);
    SignalGeneratorProperties signalGeneratorProperties(&sgViewer);
    BreadboardProperties breadboardProperties(&breadboardViewer);
    MultimeterProperties multimeterProperties(&multimeterViewer);

    DataSource dataSource(&viewer, &sgViewer, &breadboardViewer);

	scopeProperties.dataSource = &dataSource;
	dataSource.scopeProperties = &scopeProperties;
    dataSource.breadboardProperties = &breadboardProperties;
    dataSource.multimeterProperties = &multimeterProperties;

    signalGeneratorProperties.dataSource = &dataSource;
    dataSource.signalGeneratorProperties = &signalGeneratorProperties;
    QObject::connect(&signalGeneratorProperties, &SignalGeneratorProperties::changed, &dataSource, &DataSource::sgChanged);
    QObject::connect(&multimeterProperties, &MultimeterProperties::changed, &dataSource, &DataSource::sgChanged);

    multimeterProperties.dataSource = &dataSource;
    multimeterProperties.signalGeneratorProperties = &signalGeneratorProperties;

    viewer.rootContext()->setContextProperty("APP_VERSION", xstr(APP_VERSION));
	viewer.rootContext()->setContextProperty("datasource", &dataSource);
	viewer.rootContext()->setContextProperty("scopeProperties", &scopeProperties);
    viewer.rootContext()->setContextProperty("breadboardProperties", &breadboardProperties);

    sgViewer.rootContext()->setContextProperty("datasource", &dataSource);
    sgViewer.rootContext()->setContextProperty("signalGeneratorProperties", &signalGeneratorProperties);

    breadboardViewer.rootContext()->setContextProperty("breadboardProperties", &breadboardProperties);

    multimeterViewer.rootContext()->setContextProperty("multimeterProperties", &multimeterProperties);

	viewer.setSource(QUrl("qrc:/main.qml"));
	viewer.setResizeMode(QQuickView::SizeRootObjectToView);
    viewer.show();

    sgViewer.setSource(QUrl("qrc:/signal_generator.qml"));
    sgViewer.setResizeMode(QQuickView::SizeRootObjectToView);
    sgViewer.setColor("#fafafa");
    sgViewer.show();

    #ifdef QT_DEBUG
    qputenv("QTWEBENGINE_REMOTE_DEBUGGING", QByteArray::number(9000));
    #endif

    breadboardViewer.setSource(QUrl("qrc:/breadboard.qml"));
    breadboardViewer.setResizeMode(QQuickView::SizeRootObjectToView);
    breadboardViewer.show();

    multimeterViewer.setSource(QUrl("qrc:/multimeter.qml"));
    multimeterViewer.setResizeMode(QQuickView::SizeRootObjectToView);
    multimeterViewer.show();



	return app.exec();
}
