/*
Breaboard simulator
Copyright (C) 2020  Balázs Dura-Kovács

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#include "datasource.h"
#include <QtCharts/QXYSeries>
#include <QtCharts/QAreaSeries>
#include <QtQuick/QQuickView>
#include <QtQuick/QQuickItem>
#include <QFileDialog>
#include <QStringList>
#include <QEventLoop>
#include <QJsonObject>
#include <random>


QT_CHARTS_USE_NAMESPACE

Q_DECLARE_METATYPE(QAbstractSeries *)
Q_DECLARE_METATYPE(QAbstractAxis *)

DataSource::DataSource(QQuickView *appViewer, QQuickView *sgViewer, QQuickView *bbViewer, QObject *parent) :
	QObject(parent),
    m_appViewer(appViewer),
    m_sgViewer(sgViewer),
    m_bbViewer(bbViewer)
{
	qRegisterMetaType<QAbstractSeries*>();
	qRegisterMetaType<QAbstractAxis*>();

    spice = new Spice;
    spice->appViewer = appViewer;

    t.resize(SAMPLES_PER_FRAME);
    V1.resize(SAMPLES_PER_FRAME);
    V2.resize(SAMPLES_PER_FRAME);
}

void DataSource::start(){
    real_timer.start();
    triggerPossible = true;

    // get breadboard info
    QEventLoop loop;
    connect(this->breadboardProperties, &BreadboardProperties::changed, &loop, &QEventLoop::quit); //wait for JS
    breadboardProperties->requestSpice();
    loop.exec();

    QJsonObject info = breadboardProperties->info;

    /*QString circuit = R"SP1CE(test array
V1 1 0 __SIGNALGENERATOR__
R1 1 2 1500
L1 2 3 36m
C1 3 0 2700p ic=0
.tran __TIMESTEP__s __TEND__s __TSTART__s uic
.end)SP1CE";*/

    QString circuit = breadboardProperties->getSpiceCode();
    circuit += "\n.tran __TIMESTEP__s __TEND__s __TSTART__s uic\n.end";



    circuit = circuit.replace("__SIGNALGENERATOR__",signalGeneratorProperties->toSpice(breadboardProperties->student_number));

    QStringList wanted_measurements = {NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL};
    QJsonObject channel;
    QStringList channel_names = {"CH1", "CH2", "EXT"};
    for(int i = 0; i<channel_names.length(); i++){
        channel = breadboardProperties->info["oscilloscope"].toObject()[channel_names[i]].toObject();
        if(channel["connected"].toBool()){
            wanted_measurements[2*i]   = channel["nodeA"].toString();
            wanted_measurements[2*i+1] = channel["nodeB"].toString();
        }
    }
    QJsonObject voltmeter = breadboardProperties->info["multimeter"].toObject()["voltmeter"].toObject();
    spice->voltMeter = false;
    if(voltmeter["connected"].toBool()){
        spice->voltMeter = true;
        wanted_measurements[6] = voltmeter["nodeA"].toString();
        wanted_measurements[7] = voltmeter["nodeB"].toString();
    }

    //qDebug() << wanted_measurements;


    spice->start(circuit, (qreal)scopeProperties->secondsPdiv * 10. / SAMPLES_PER_FRAME, signalGeneratorProperties->realFrequency , wanted_measurements);
}

void DataSource::stop(){
    spice->pause();
}

void DataSource::resume(){
    spice->resume();
}

void DataSource::sgChanged(){
    QObject * runButton = m_appViewer->findChild<QObject*>("runButton");
    if(runButton != NULL && runButton->property("running").toBool()){
        this->start();
    }
}

void DataSource::triggerChanged(){
    triggerPossible = true;
}


void DataSource::update(QAbstractSeries * ch1, QAbstractSeries * ch2, QAbstractSeries * xy){

    if(rendering || !spice->buffer_ready){
        return;
    }
    rendering = true;

    multimeterProperties->setValue(spice->rmsVoltage2,breadboardProperties->student_number);

    qreal timestep = (qreal)scopeProperties->secondsPdiv * 10. / SAMPLES_PER_FRAME;
    qreal real_time = (qreal)real_timer.nsecsElapsed() * 1e-9;
    real_time = round(real_time*30.)/30.; // round to nearest 1/30 second
    const size_t buffer_size = spice->buffer_size;
    qreal intpart;
    qreal phase = modf(real_time / ((qreal)buffer_size*timestep),&intpart);

    size_t start_sample = (qreal)buffer_size * phase;

    qreal totalTime = scopeProperties->secondsPdiv * 10;
	qreal t_start = scopeProperties->t_offset - scopeProperties->secondsPdiv * 5;

    qreal v1_max = scopeProperties->V1_offset + scopeProperties->voltsPdiv1 * 5;
    qreal v1_min = scopeProperties->V1_offset - scopeProperties->voltsPdiv1 * 5;
    qreal v1_range = scopeProperties->voltsPdiv1 * 10;

    qreal v2_max = scopeProperties->V2_offset + scopeProperties->voltsPdiv2 * 5;
    qreal v2_min = scopeProperties->V2_offset - scopeProperties->voltsPdiv2 * 5;
    qreal v2_range = scopeProperties->voltsPdiv2 * 10;

    size_t offset = start_sample;



	const int resolution = 1<<11;
    size_t bin_scale1 = resolution / v1_range;
    size_t bin_scale2 = resolution / v2_range;

    // trigger
    if(!triggerPossible || scopeProperties->triggerMode == "OFF"){
        goto TRIGEND;
    }

    {

    triggerPossible = true;
    QVector<qreal> * triggerChannel;
    if(scopeProperties->triggerMode == "CH1"){
        triggerChannel = &spice->CH1;
    }else if(scopeProperties->triggerMode == "CH2"){
        triggerChannel = &spice->CH2;
    }else if(scopeProperties->triggerMode == "EXT"){
        triggerChannel = &spice->EXT;
    }else{
        goto TRIGEND;
    }

    size_t trigger_offset;
    qreal last_value = triggerChannel->at((offset-1)%buffer_size);
    qreal new_value;
    qreal level = scopeProperties->triggerLevel;
    if(scopeProperties->triggerEdge == "/"){
        for(trigger_offset = 0; trigger_offset<buffer_size; trigger_offset++){
            new_value = triggerChannel->at((offset+trigger_offset)%buffer_size);
            if(last_value < level && new_value >= level){
                offset += trigger_offset;
                goto TRIGEND;
            }
            last_value = new_value;
        }
    }else{
        for(trigger_offset = 0; trigger_offset<buffer_size; trigger_offset++){
            new_value = triggerChannel->at((offset+trigger_offset)%buffer_size);
            if(last_value > level && new_value <= level){
                offset += trigger_offset;
                goto TRIGEND;
            }
            last_value = new_value;
        }
    }

    triggerPossible = false;

    }



TRIGEND:

    //add offset oscilloscope
    offset += SAMPLES_PER_FRAME / 2 + ((scopeProperties->t_offset / totalTime) * (qreal)SAMPLES_PER_FRAME);

    for(size_t i=0; i<SAMPLES_PER_FRAME; i++){
        t[i] = t_start + totalTime * qreal(i) / SAMPLES_PER_FRAME;

        V1[i] = spice->CH1[(i+offset)%buffer_size] + gauss_distribution(generator);
        V2[i] = spice->CH2[(i+offset)%buffer_size] + gauss_distribution(generator);

        // saturating signal
        if(V1[i]>v1_max){
            V1[i] = v1_max;
        }else if(V1[i]<v1_min){
            V1[i] = v1_min;
		}
        if(V2[i]>v2_max){
            V2[i] = v2_max;
        }else if(V2[i]<v2_min){
            V2[i] = v2_min;
        }

		// simulate 12bit ADC binning
        V1[i] = ((qreal)((int)(bin_scale1 * (V1[i] - v1_min))))/bin_scale1 + v1_min;
        V2[i] = ((qreal)((int)(bin_scale2 * (V2[i] - v2_min))))/bin_scale2 + v2_min;
	}

    draw(ch1, ch2, xy);
    rendering = false;
}

void DataSource::draw(QAbstractSeries * ch1, QAbstractSeries * ch2, QAbstractSeries * xy){
    if(scopeProperties->xyMode){
        QVector<QPointF> xyPoints(SAMPLES_PER_FRAME);
        for(size_t i=0; i<SAMPLES_PER_FRAME; i++){
            xyPoints[i] = {V1[i]/scopeProperties->V1_scale,  V2[i]/scopeProperties->V2_scale};
        }
        QXYSeries *xyXY = static_cast<QXYSeries *>(xy);
        xyXY->replace(xyPoints);

        return;
    }

    // scale to scope
    QVector<QPointF> ch1Points(SAMPLES_PER_FRAME);
    QVector<QPointF> ch2Points(SAMPLES_PER_FRAME);

    for(size_t i=0; i<SAMPLES_PER_FRAME; i++){
        ch1Points[i] = {t[i]/scopeProperties->t_scale, V1[i]/scopeProperties->V1_scale};
        ch2Points[i] = {t[i]/scopeProperties->t_scale, V2[i]/scopeProperties->V2_scale};
	}



	QXYSeries *xych1 = static_cast<QXYSeries *>(ch1);
    QXYSeries *xych2 = static_cast<QXYSeries *>(ch2);
    xych1->replace(ch1Points);
    xych2->replace(ch2Points);
}

void DataSource::saveCSV(){
    QByteArray csv = "time(s),CH1(V),CH2(V)";
	QString line;
    for(size_t i=0; i<SAMPLES_PER_FRAME; i++){
        line = QString::asprintf("\n%.5g,%.5g,%.5g", t[i], V1[i], V2[i]);
        csv.append(line.toUtf8());
	}
	QFileDialog::saveFileContent(csv, "acquisition.csv");
}
