/*
Breaboard simulator
Copyright (C) 2020  Balázs Dura-Kovács

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#include <QtQuick/QQuickView>
#include <QFile>
#include <QVariantMap>
#include <random>

#include "breadboardproperties.h"
#include "siphash.h"




BreadboardProperties::BreadboardProperties(QQuickView *bbViewer, QObject *parent) :
	QObject(parent),
    m_viewer(bbViewer)
{
    

}

void BreadboardProperties::requestSpice(){
    QObject * communicator = m_viewer->findChild<QObject*>("communicator");
    QMetaObject::invokeMethod(communicator, "requestSpice");
}

void BreadboardProperties::enable(bool enabled){
    QObject * webView = m_viewer->findChild<QObject*>("web");
    webView->setProperty("enabled", enabled);
    QObject * communicator = m_viewer->findChild<QObject*>("communicator");
    communicator->setProperty("webenabled", enabled);
    QMetaObject::invokeMethod(communicator, "toggleEnabled");
}

void BreadboardProperties::update(QJsonObject info){
    this->info = info;
    emit changed();
}

QString BreadboardProperties::getSpiceCode(){
    //qDebug() << info;
    QString circuit = info["spice"].toString();
    student_number = info["studentnumber"].toString();
    QVariantMap form = info.toVariantMap()["formData"].toMap();

    const uint8_t k[16] = {3, 1, 4, 1, 5, 9, 2, 6, 5, 3, 5, 8, 9, 7, 9, 3};
    std::default_random_engine generator;

    QVariantMap::const_iterator i = form.begin();
    while(i != form.constEnd()) {
        QVariantMap properties = i.value().toMap();

        QString name = i.key();
        qreal value = properties["value"].toDouble();
        qreal tolerance = properties["tolerance"].toDouble();
        QString suffix = properties["suffix"].toString();

        // randomise value within tolerance deterministically
        QString toHash = name + QString::number(value,'f',15) + suffix + QString::number(tolerance,'f',15) + student_number;
        const std::string& toHashStd = toHash.toStdString();

        qint64 hash = 0;
        uint8_t hash_s[8];
        siphash((uint8_t * )toHashStd.c_str(), toHashStd.length(), k, (uint8_t *) hash_s, 8);
        memcpy(&hash, hash_s, 8);

        generator.seed(hash);
        std::normal_distribution<qreal> gauss_distribution = std::normal_distribution<qreal>(1.,tolerance/300.);
        value = value * gauss_distribution(generator);

        circuit = circuit.replace("__" + name + "_VALUE__",QString::number(value,'f',15));

        i++;

    }

    return circuit;
}

QByteArray BreadboardProperties::loadBbSvg(){
    QFile file(":/breadboard/svg/breadboard2.svg");
    file.open(QIODevice::ReadOnly);

    return file.readAll();
}
