/*
Breaboard simulator
Copyright (C) 2020  Balázs Dura-Kovács

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

import QtQuick 2.15
import QtQuick.Layouts 1.11
import QtQuick.Controls 2.15
import QtQuick.Controls.Styles 1.4


Frame{
    visible: true

    id: signalGenerator
    objectName: "signalGenerator"

    readonly property variant waveForms: ["SIN", "SQUARE", "TRIANGLE", "SAWTOOTH"]

    property string frequency: '1000.0'
    property double dcOffset: 0.0
    property double amplitude: 1.0
    property string waveform: "SIN"


    function updateProperties(){
        this.frequency = frequency.text;
        this.dcOffset = dcOffset.value;
        this.amplitude = amplitudeDial.value;
        if(amplitudeAttenuated.checked){
            this.amplitude *= 0.1
        }
        signalGeneratorProperties.update();
    }


    GridLayout {
        columns: 3
        rows: 4

        Text{
            text: "Frequency"
        }

        RowLayout{
            Dial {
                id: frequencyDial
                wheelEnabled: true
                inputMode: "Circular"
                from: -1.0
                to: Math.log(4e6)/Math.log(10)
                stepSize: 0.01
                value: Math.log(1000.0)/Math.log(10)
                ToolTip.text: "Coarse"
                ToolTip.visible: hovered
                onMoved: {
                    frequency.text = signalGeneratorProperties.formatFrequency(Math.round(Math.pow(10.0, this.value + frequencyFineTuner.value) * 10) / 10)
                    frequencyFineTuner.value = 0;
                    signalGenerator.updateProperties();
                }

                Layout.maximumWidth: 60
                Layout.alignment: Qt.AlignHCenter
            }

            Dial {
                id: frequencyFineTuner
                wheelEnabled: true
                inputMode: "Circular"
                from: -0.05
                to: 0.05
                stepSize: 0.0001
                value: 0
                ToolTip.text: "Fine"
                ToolTip.visible: hovered
                onMoved: {
                    frequency.text = signalGeneratorProperties.formatFrequency(Math.round(Math.pow(10.0, this.value + frequencyDial.value) * 10) / 10)
                    signalGenerator.updateProperties();
                }
                Layout.maximumWidth: 60
                Layout.alignment: Qt.AlignHCenter
            }
        }

        RowLayout{
            TextInput {
                id: frequency
                text: '1000.0'
                overwriteMode: false
                selectByMouse: true
                horizontalAlignment: Text.AlignRight
                Layout.minimumWidth: 150
                validator: DoubleValidator {
                    bottom: 0.1
                    top: 4000000.0
                    decimals: 1
                    notation: DoubleValidator.StandardNotation
                    locale: 'en_GB'
                }
                onTextEdited: {
                    frequencyDial.value = Math.log(parseFloat(this.text))/Math.log(10);
                    frequencyFineTuner.value = 0;
                    signalGenerator.updateProperties();
                }
            }

            Text{
                text: "Hz"
            }
        }

        Text{
            text: "DC offset"
        }

        Dial {
            id: dcOffset
            wheelEnabled: true
            inputMode: "Circular"
            from: -5.0
            to: 5.0
            stepSize: 0.01
            value: 0
            Layout.maximumWidth: 60
            Layout.alignment: Qt.AlignHCenter
            onMoved: {
                signalGenerator.updateProperties();
            }
        }

        Button {
            text: "Reset"
            Layout.alignment: Qt.AlignHCenter
            onClicked: {
                dcOffset.value = 0;
                signalGenerator.updateProperties();
            }
        }

        Text{
            text: "Amplitude"
        }

        Dial {
            id: amplitudeDial
            wheelEnabled: true
            inputMode: "Circular"
            from: 0.01
            to: 5.0
            value: 1.0
            stepSize: 0.01
            Layout.maximumWidth: 60
            Layout.alignment: Qt.AlignHCenter
            onMoved: {
                signalGenerator.updateProperties();
            }
        }

        Switch {
            id: amplitudeAttenuated
            checked: false
            text: "-20dB"
            Layout.alignment: Qt.AlignHCenter
            onCheckedChanged: {
                signalGenerator.updateProperties();
            }
        }

        Text{
            text: "Waveform"
        }

        ComboBox {
            id: waveFormCombo
            model: signalGenerator.waveForms
            currentIndex: 0
            Layout.columnSpan: 2
            Layout.fillWidth: true
            onCurrentIndexChanged: {
                signalGenerator.waveform = signalGenerator.waveForms[currentIndex];
                signalGenerator.updateProperties();
            }

        }




    }

}
