/*
Breaboard simulator
Copyright (C) 2020  Balázs Dura-Kovács

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

import QtQuick 2.15
import QtCharts 2.3
import QtQuick.Layouts 1.11
import QtQuick.Controls 2.15
import QtQuick.Dialogs 1.3


Frame{
    visible: true




    RowLayout {

        id: scope

        readonly property variant timeDivisionsText: ["10 ns", "20 ns", "50 ns", "100 ns", "200 ns", "500 ns",
            "1 μs", "2 μs", "5 μs", "10 μs", "20 μs", "50 μs", "100 μs", "200 μs", "500 μs",
            "1 ms", "2 ms", "5 ms", "10 ms", "20 ms", "50 ms", "100 ms", "200 ms", "500 ms",
            "1 s", "2 s", "5 s"]
        readonly property variant timeDivisions: [10e-9, 20e-9, 50e-9, 100e-9, 200e-9, 500e-9,
            1e-6, 2e-6, 5e-6, 10e-6, 20e-6, 50e-6, 100e-6, 200e-6, 500e-6,
            1e-3, 2e-3, 5e-3, 10e-3, 20e-3, 50e-3, 100e-3, 200e-3, 500e-3,
            1., 2., 5.]
        readonly property variant timeDivisionUnits: ["ns", "ns", "ns", "ns", "μs", "μs",
            "μs", "μs", "μs", "μs", "μs", "μs", "μs", "ms", "ms",
            "ms", "ms", "ms", "ms", "ms", "ms", "ms", "s", "s",
            "s", "s", "s"]
        readonly property variant voltDivisionsText: ["100 μV", "200 μV", "500 μV",
            "1 mV", "2 mV", "5 mV", "10 mV", "20 mV", "50 mV", "100 mV", "200 mV", "500 mV",
            "1 V", "2 V", "5 V"]
        readonly property variant voltDivisions: [100e-6, 200e-6, 500e-6,
            1e-3, 2e-3, 5e-3, 10e-3, 20e-3, 50e-3, 100e-3, 200e-3, 500e-3,
            1., 2., 5.]
        readonly property variant voltDivisionUnits: ["μV", "mV", "mV",
            "mV", "mV", "mV", "mV", "mV", "mV", "mV", "V", "V",
            "V", "V", "V"]
        readonly property variant triggerModes: ["OFF", "CH1", "CH2", "EXT"]
        readonly property variant triggerEdges: ["/", "\\"]

        function updateProperties(){
            if(runButton.firstclick){
                scopeProperties.init();
                runButton.firstclick = false;
            }else{
                scopeProperties.update();
            }

            datasource.draw(scopeView.series(0), scopeView.series(1), scopeView.series(2));
        }


        ChartView {
             id: scopeView
             objectName: "scopeView"
             width: 900
             height: 900

             animationOptions: ChartView.NoAnimation
             theme: ChartView.ChartThemeBlueNcs
             antialiasing: true
             legend.visible: false


             ValueAxis {
                 property double secondsPdiv: 0.001
                 property double scaleUnit: 0.001 // eg. if it is 0.001, then scale will be in units of ms
                 property double offset: 0.0 //scale centre in s

                 id: axisT1
                 objectName: "axisT1"

                 min: -5
                 max: +5
                 titleText: "Time (ms)"
                 tickType: ValueAxis.TicksFixed
                 tickAnchor: 0.
                 tickCount: 11
                 gridVisible: true
                 minorGridVisible: true
                 minorGridLinePen: scopeProperties.getMinorGridPen(scopeView.plotArea.height)
                 minorTickCount: 4
                 labelFormat: "%5.3g"
                 labelsFont.pointSize: 5

                 visible: !xySwitch.checked

                 function updateSecondsPdiv(index){
                     let unit = scope.timeDivisionUnits[index];
                     titleText = "Time (" + unit + ")";

                     const unitMap = {"ns":1e-9, "μs": 1e-6, "ms": 1e-3, "s": 1.};

                     secondsPdiv = scope.timeDivisions[index];
                     scaleUnit = unitMap[unit];

                     min = (-secondsPdiv*5 + offset)/scaleUnit
                     max = (+secondsPdiv*5 + offset)/scaleUnit
                     //labelFormat = max < 2.6 ? "%1.1f" : "%1.0f";

                 }
             }

             ValueAxis {
                 property double voltsPdiv: 1.0
                 property double scaleUnit: 1.0 // eg. if it is 0.001, then scale will be in units of mV
                 property double offset: 0.0 //scale centre in V

                 id: axisY1
                 objectName: "axisY1"

                 min: -5
                 max: 5
                 titleText: "CH1 V"
                 minorGridVisible: true
                 minorGridLinePen: scopeProperties.getMinorGridPen(scopeView.plotArea.width)
                 minorTickCount: 4
                 gridVisible: true
                 tickType: ValueAxis.TicksFixed
                 tickAnchor: 0.
                 tickCount: 11
                 labelFormat: "%6.3f"
                 labelsFont.pointSize: 5
                 color: "#9E0000"

                 visible: !xySwitch.checked

                 function updateVoltsPdiv(index){
                     let unit = scope.voltDivisionUnits[index];
                     titleText = `CH1 (${unit})`;

                     const unitMap = {"μV": 1e-6, "mV": 1e-3, "V": 1.};

                     voltsPdiv = scope.voltDivisions[index];
                     scaleUnit = unitMap[unit];

                     min = (-voltsPdiv*5 + offset)/scaleUnit
                     max = (+voltsPdiv*5 + offset)/scaleUnit
                     //labelFormat = max < 2.6 ? "%1.1f" : "%1.0f";
                 }
             }


             ValueAxis {
                 property double voltsPdiv: 1.0
                 property double scaleUnit: 1.0 // eg. if it is 0.001, then scale will be in units of mV
                 property double offset: 0.0 //scale centre in V

                 id: axisY2
                 objectName: "axisY2"

                 min: -5
                 max: 5
                 titleText: "CH2 V"
                 minorGridVisible: false
                 gridVisible: true
                 tickType: ValueAxis.TicksFixed
                 tickAnchor: 0.
                 tickCount: 11
                 labelFormat: "%6.3f"
                 labelsFont.pointSize: 5
                 color: "#060089"

                 visible: !xySwitch.checked


                 function updateVoltsPdiv(index){
                     let unit = scope.voltDivisionUnits[index];
                     titleText = `CH2 (${unit})`;

                     const unitMap = {"μV": 1e-6, "mV": 1e-3, "V": 1.};

                     voltsPdiv = scope.voltDivisions[index];
                     scaleUnit = unitMap[unit];

                     min = (-voltsPdiv*5 + offset)/scaleUnit
                     max = (+voltsPdiv*5 + offset)/scaleUnit
                     //labelFormat = max < 2.6 ? "%1.1f" : "%1.0f";
                 }
             }

             ValueAxis {
                 property double voltsPdiv: axisY1.voltsPdiv
                 property double scaleUnit: axisY1.scaleUnit
                 property double offset: axisY1.offset

                 id: axisXY_X
                 objectName: "axisXY_X"

                 min: axisY1.min
                 max: axisY1.max
                 titleText: axisY1.titleText
                 minorGridVisible: true
                 minorGridLinePen: scopeProperties.getMinorGridPen(scopeView.plotArea.height)
                 minorTickCount: 4
                 tickType: axisY1.tickType
                 tickAnchor: axisY1.tickAnchor
                 tickCount: axisY1.tickCount
                 labelFormat: axisY1.labelFormat
                 labelsFont.pointSize: axisY1.labelsFont.pointSize

                 visible: xySwitch.checked
             }

             ValueAxis {
                 property double voltsPdiv: axisY2.voltsPdiv
                 property double scaleUnit: axisY2.scaleUnit
                 property double offset: axisY2.offset

                 id: axisXY_Y
                 objectName: "axisXY_Y"

                 min: axisY2.min
                 max: axisY2.max
                 titleText: axisY2.titleText
                 minorGridVisible: true
                 minorGridLinePen: scopeProperties.getMinorGridPen(scopeView.plotArea.width)
                 minorTickCount: 4
                 gridVisible: axisY2.gridVisible
                 tickType: axisY2.tickType
                 tickAnchor: axisY2.tickAnchor
                 tickCount: axisY2.tickCount
                 labelFormat: axisY2.labelFormat
                 labelsFont.pointSize: axisY2.labelsFont.pointSize

                 visible: xySwitch.checked
             }

             LineSeries {
                 id: ch1
                 objectName: "ch1"
                 property bool on: ch1Switch.checked && !xySwitch.checked

                 axisX: axisT1
                 axisY: axisY1
                 useOpenGL: true
                 width: 3
                 color: "#9E0000"
                 visible: on
             }

             LineSeries {
                 id: ch2
                 objectName: "ch2"
                 property bool on: ch2Switch.checked && !xySwitch.checked

                 axisX: axisT1
                 axisYRight: axisY2
                 useOpenGL: true
                 width: 3
                 color: "#060089"
                 visible: on

             }

             LineSeries {
                 id: xy
                 objectName: "xy"
                 property bool on: xySwitch.checked

                 axisX: axisXY_X
                 axisY: axisXY_Y
                 useOpenGL: true
                 width: 3
                 color: "#4E9A06"
                 visible: on

             }
         }


        ColumnLayout {
            id: controlPanel
            x: 400
            y: 111

            spacing: 12

            Layout.fillHeight: true

            GridLayout {
                rows: 3
                columns: 2

                Text {
                    text: "Time/div"
                }

                ComboBox {
                    id: timedivCombo
                    Layout.minimumWidth: 200
                    Layout.fillWidth: true
                    currentIndex: 15
                    model: scope.timeDivisionsText
                    onCurrentIndexChanged: {
                        axisT1.updateSecondsPdiv(currentIndex);
                        tOffsetDial.update();
                        if(runButton.running){
                            datasource.start();
                        }
                    }

                }

                Text {
                    text: "CH1 Volts/div"
                }

                ComboBox {
                    id: ch1VoltdivCombo
                    Layout.minimumWidth: 200
                    Layout.fillWidth: true
                    currentIndex: 12
                    model: scope.voltDivisionsText
                    onCurrentIndexChanged: {
                        axisY1.updateVoltsPdiv(currentIndex);
                        ch1OffsetDial.update();
                    }
                }

                Text {
                    text: "CH2 Volts/div"
                }

                ComboBox {
                    id: ch2VoltdivCombo
                    Layout.minimumWidth: 200
                    Layout.fillWidth: true
                    currentIndex: 12
                    model: scope.voltDivisionsText
                    onCurrentIndexChanged: {
                        axisY2.updateVoltsPdiv(currentIndex);
                        ch2OffsetDial.update();
                    }
                }
            }

            RowLayout{
                Button {
                    id: runButton
                    objectName: "runButton"
                    property bool running: false
                    property bool firstclick: true
                    text: running ? "■ Stop" : "▶ Run"
                    onClicked: {
                        if(firstclick){
                            scopeProperties.init();
                            firstclick = false;

                        }

                        if(!running){
                            datasource.start();
                            breadboardProperties.enable(false);
                        }else{
                            breadboardProperties.enable(true);
                        }


                        running = !running;
                    }
                }

                Button {
                    text: "Save CSV"
                    onClicked: {
                        datasource.saveCSV();
                    }
                }




            }



            GridLayout {
                rows: 6
                columns: 2

                Text {
                    text: "CH1 offset"
                    Layout.alignment: Qt.AlignHCenter
                }

                Text {
                    text: "Time offset"
                    Layout.alignment: Qt.AlignHCenter
                }

                Dial {
                    id: ch1OffsetDial // offset in units of division
                    wheelEnabled: true
                    Layout.fillWidth: true
                    Layout.alignment: Qt.AlignHCenter
                    inputMode: "Circular"
                    from: -5
                    to: 5
                    stepSize: 0.1
                    onMoved: update()
                    function update() {
                        axisY1.offset = axisY1.voltsPdiv * ch1OffsetDial.value;
                        axisY1.updateVoltsPdiv(ch1VoltdivCombo.currentIndex);
                        scope.updateProperties();
                    }
                }

                Dial {
                    id: tOffsetDial // offset in units of division
                    wheelEnabled: true
                    Layout.fillWidth: true
                    Layout.alignment: Qt.AlignHCenter
                    inputMode: "Circular"
                    from: -5
                    to: 5
                    stepSize: 0.1
                    onMoved: update()
                    function update() {
                        axisT1.offset = axisT1.secondsPdiv * tOffsetDial.value;
                        axisT1.updateSecondsPdiv(timedivCombo.currentIndex);
                        scope.updateProperties();
                    }
                }

                Button {
                    text: "Reset"
                    Layout.alignment: Qt.AlignHCenter
                    onClicked: {
                        ch1OffsetDial.value = 0;
                        ch1OffsetDial.update();
                    }
                }

                Button {
                    text: "Reset"
                    Layout.alignment: Qt.AlignHCenter
                    onClicked: {
                        tOffsetDial.value = 0;
                        tOffsetDial.update();
                    }
                }

                Text {
                    text: "CH2 offset"
                    Layout.alignment: Qt.AlignHCenter
                }

                ColumnLayout {
                    Layout.rowSpan: 3

                    Switch {
                        id: ch1Switch
                        checked: true
                        text: 'CH 1'
                        Layout.alignment: Qt.AlignHCenter
                    }

                    Switch {
                        id: ch2Switch
                        checked: false
                        text: 'CH 2'
                        Layout.alignment: Qt.AlignHCenter
                    }

                    Switch {
                        id: xySwitch
                        objectName: "xySwitch"
                        checked: false
                        text: '   XY'
                        Layout.alignment: Qt.AlignHCenter
                        enabled: ch1Switch.checked && ch2Switch.checked
                        onEnabledChanged: {
                            if(!xySwitch.enabled){
                                xySwitch.checked = false;
                            }
                        }
                        onCheckedChanged: {
                            scope.updateProperties();
                        }
                    }

                    RowLayout{
                        id: trigger
                        objectName: "trigger"
                        property string mode: "OFF"
                        property string edge: "/"
                        property double level: 0.0
                        Layout.alignment: Qt.AlignHCenter
                        Text {
                            text: 'TRIG'
                        }

                        ComboBox {
                            id: triggerCombo
                            model: scope.triggerModes
                            currentIndex: 0
                            ToolTip.text: "Source"
                            ToolTip.visible: hovered
                            onCurrentIndexChanged: {
                                trigger.mode = scope.triggerModes[currentIndex];
                                scope.updateProperties();
                            }
                        }

                        ComboBox {
                            id: triggerEdgeCombo
                            model: scope.triggerEdges
                            currentIndex: 0
                            ToolTip.text: "Edge (rising/falling)"
                            ToolTip.visible: hovered
                            onCurrentIndexChanged: {
                                trigger.edge = scope.triggerEdges[currentIndex];
                                scope.updateProperties();
                            }
                        }
                    }

                    TextInput {
                        id: triggerLevelText
                        text: '0.0'
                        overwriteMode: false
                        selectByMouse: true
                        horizontalAlignment: Text.AlignRight
                        Layout.minimumWidth: 150
                        Layout.fillWidth: true
                        ToolTip.text: "Trigger level"
                        ToolTip.visible: hovered
                        validator: DoubleValidator {
                            bottom: -10
                            top: 10
                            decimals: 5
                            notation: DoubleValidator.StandardNotation
                            locale: 'en_GB'
                        }
                        onTextEdited: {
                            trigger.level = this.text
                            scope.updateProperties();
                        }
                    }





                }

                Dial {
                    id: ch2OffsetDial // offset in units of division
                    wheelEnabled: true
                    Layout.fillWidth: true
                    inputMode: "Circular"
                    from: -5
                    to: 5
                    stepSize: 0.1
                    onMoved: update()
                    function update() {
                        axisY2.offset = axisY2.voltsPdiv * ch2OffsetDial.value;
                        axisY2.updateVoltsPdiv(ch2VoltdivCombo.currentIndex);
                        scope.updateProperties();
                    }
                }

                Button {
                    text: "Reset"
                    Layout.alignment: Qt.AlignHCenter
                    onClicked: {
                        ch2OffsetDial.value = 0;
                        ch2OffsetDial.update();
                    }
                }


            }


       }


    }


    Timer {
        id: refreshTimer
        interval: Math.max(1 / 30. * 1000, axisT1.secondsPdiv*10.*1000) // max 30 Hz
        running: runButton.running
        repeat: true
        onTriggered: {
            datasource.update(scopeView.series(0), scopeView.series(1), scopeView.series(2));
            // TODO: pause timer while updating
            //console.log(typeof(scopeView))
        }
    }

    MessageDialog {
        id: updateDialog
        property string url: "https://www.homepages.ucl.ac.uk/~zcapbmk/breadboard/"
        property bool opened: false
        title: "Update available"
        text: "A new version of the breaboard simulator is available on the following URL: " + this.url
        standardButtons: StandardButton.Open | StandardButton.Cancel
        onAccepted: {
            if(!opened){
                Qt.openUrlExternally(this.url);
                opened = true;
            }
        }
        onRejected: {

        }
    }

    Component.onCompleted : {
        var xhr = new XMLHttpRequest();
        var url = "http://www.homepages.ucl.ac.uk/~zcapbmk/breadboard/version.txt";

        xhr.onreadystatechange = function() {
            if (xhr.readyState === 4 && xhr.status === 200) {
                if(versionCompare(xhr.responseText, APP_VERSION) > 0) {
                    updateDialog.open()
                }
            }
        }
        xhr.open('GET', url, true);
        xhr.send('');
    }

    function versionCompare(a, b) {
        if (a === b) {
           return 0;
        }

        var a_components = a.split(".");
        var b_components = b.split(".");

        var len = Math.min(a_components.length, b_components.length);

        // loop while the components are equal
        for (var i = 0; i < len; i++) {
            // A bigger than B
            if (parseInt(a_components[i]) > parseInt(b_components[i])) {
                return 1;
            }

            // B bigger than A
            if (parseInt(a_components[i]) < parseInt(b_components[i])) {
                return -1;
            }
        }

        // If one's a prefix of the other, the longer one is greater.
        if (a_components.length > b_components.length) {
            return 1;
        }

        if (a_components.length < b_components.length) {
            return -1;
        }

        // Otherwise they are the same.
        return 0;
    }


}

