/*
Virtual breadboard
Copyright (C) 2020  Balázs Dura-Kovács

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <https://www.gnu.org/licenses/>.
*/

function noop(){
	return null;
}

const compass = {
	NORTH: 3,
	EAST: 0,
	SOUTH: 1,
	WEST: 2
};

class Circuit{
	constructor(){
		this.elementCounter = 0;
		this.phPinCounter = 0;
		this.graph = new graphlib.Graph({directed: false}); // nodes are physical pins on the same potential (connected with bare metal)
		this.graphAllPot = new graphlib.Graph({directed: false}); // nodes are not necessarliy on the same potential, two ends of a resistor are connected for example
		this.elements = [];
		this.pinSelectA = null; //type Pin
		this.phGround = null; 	// physical pin connected to ground
								// signal generator's out currently

		// add breadboard pins
		let column, row;
		let pinIds;
		for(column = 1; column<=63; column++){
			pinIds = [];
			for(row='A'.charCodeAt(0); row<='E'.charCodeAt(0); row++){
				let pinId = 'pin' + column + String.fromCharCode(row);
				pinIds.push(pinId);
			}
			this.newElement('breadboardGroup', pinIds);

			pinIds = [];
			for(row='F'.charCodeAt(0); row<='J'.charCodeAt(0); row++){
				let pinId = 'pin' + column + String.fromCharCode(row);
				pinIds.push(pinId);
			}
			this.newElement('breadboardGroup', pinIds);
		}
		//power rails
		for(row='W'.charCodeAt(0); row<='Z'.charCodeAt(0); row++){
			pinIds = [];
			for(column = 3; column<=61; column++){
				if(column % 6 == 2){ //gaps
					continue;
				}
				let pinId = 'pin' + column + String.fromCharCode(row);
				pinIds.push(pinId);
			}
			this.newElement('breadboardGroup', pinIds);
		}

		this.newElement('oscilloscope');
		this.newElement('signal-generator');
		this.newElement('multimeter');


		// unselect if clicked on canvas
		document.getElementById("canvas").addEventListener("click", this.unselect.bind(this));
	}

	newElement(type, options=null){
		var element;
		
		switch(type){
			case "resistor":
				element = new Resistor(this.elementCounter, this);
				break;
			case "inductor":
				element = new Inductor(this.elementCounter, this);
				break;
			case "capacitor":
				element = new Capacitor(this.elementCounter, this);
				break;
			case "breadboardGroup":
				element = new BreadboardGroup(this.elementCounter, this, options);
				break;
			case "oscilloscope":
				element = new Oscilloscope(this.elementCounter, this);
				break;
			case "signal-generator":
				element = new SignalGenerator(this.elementCounter, this);
				break;
			case "multimeter":
				element = new Multimeter(this.elementCounter, this);
				break;
			case "clips":
				element = new Clips(this.elementCounter, this);
				break;
			case "tee":
				element = new Tee(this.elementCounter, this);
				break;
			default:
				return false;
		}

		this.elementCounter++;

		this.elements.push(element);

		return element;
	}

	newPhPinId(){
		this.phPinCounter++;
		return this.phPinCounter;
	}

	// unselect pins
	unselect(event){
		if(event.target.id !== null && event.target.id == 'canvas' && this.pinSelectA){
			this.pinSelectA.domElement.classList.remove("selected");
			this.pinSelectA = null;
		}
	}

	// make conenction between two pins
	connect(pin){
		let nodeA;
		let nodeB;

		nodeA = this.pinSelectA.circuitElement.id;
		nodeB = pin.circuitElement.id;

		// don't connect if
		// pins are the same
		if(pin == this.pinSelectA){
			return;
		}

		// at least one end should be breadboard
		if(this.pinSelectA.circuitElement.isDraggable && pin.circuitElement.isDraggable){
			return;
		}

		// can't connect bnc to jumper wire
		if(this.pinSelectA.type != pin.type){
			return;
		}

		if(this.pinSelectA.phPins.length != pin.phPins.length){
			// these should always be the same, this is just a sanity check
			console.error("The number of physical pins don't match");
			return;
		}

		// connect each physical pin to the corresponding physical pin of the other connector
		// warning: physical pins should be in the same order
		for(var i=0; i<pin.phPins.length; i++){
			this.graph.setEdge(this.pinSelectA.phPins[i].id, pin.phPins[i].id);
			this.graphAllPot.setEdge(this.pinSelectA.phPins[i].id, pin.phPins[i].id);
		}

		let startSocket = this.pinSelectA.circuitElement.socketOrientation(this.pinSelectA);
		let endSocket = pin.circuitElement.socketOrientation(pin);
		startSocket = CircuitElement.rotateMapping(startSocket, this.pinSelectA.circuitElement.orientation);
		endSocket = CircuitElement.rotateMapping(startSocket, pin.circuitElement.orientation);


		let anchorA =  LeaderLine.pointAnchor(this.pinSelectA.domElement, this.pinSelectA.anchorPos());
		let anchorB =  LeaderLine.pointAnchor(pin.domElement, pin.anchorPos());

		// bnc cable is thicker than jumper wire
		let lineWidth = pin.type == 'bnc' ? 5 : 2;

		//use simpler connector line for pins on breadboard
		let path = 'fluid'; //straight/arc/fluid/magnet/grid
		if(this.pinSelectA.circuitElement.type == 'breadboard' && pin.circuitElement.type == 'breadboard'){
			path = 'arc';
		}

		let line = new LeaderLine(
			anchorA,
			anchorB,
			{
				color: 'black', size: lineWidth, path: path,
				startPlug: 'disc', endPlug: 'disc',
				startPlugSize: 0.1, endPlugSize: 0.1,
				startSocket: startSocket, endSocket: endSocket
			}
		);


		let connection = new Connection(this.pinSelectA, pin, line);
		this.pinSelectA.connection = connection;
		pin.connection = connection;

		this.pinSelectA.domElement.classList.remove("selected");
		pin.domElement.classList.remove("selected");
		this.pinSelectA = null;
	}

	disconnect(pin){
		let connection = pin.connection;
		if(connection === null){
			return;
		}
		connection.line.remove();
		delete connection.line;

		let pinA = connection.pinA;
		let pinB = connection.pinB;

		if(pinA.phPins.length != pinB.phPins.length){
			// these should always be the same, this is just a sanity check
			console.error("The number of physical pins don't match");
			return;
		}

		// disconnect each physical pin from the corresponding physical pin of the other connector
		// warning: physical pins should be in the same order
		for(var i=0; i<pin.phPins.length; i++){
			circuit.graph.removeEdge(pinA.phPins[i].id, pinB.phPins[i].id);
			circuit.graphAllPot.removeEdge(pinA.phPins[i].id, pinB.phPins[i].id);
		}

		pinA.connection = null;
		pinB.connection = null;
	}

	// generate spice code
	getSpice(){

		let that = this;
		// returns a map of physical pin id => spice node id
		function getIdMap(graph){
			// get list of connected components ("groups")
			// each group of physical pins are interconnected with bare metal
			// ie. there are on the same potential (DC approx.)
			const groups = graphlib.alg.components(graph);


			// index each physical pin: which group it belongs to
			let nodeId = []; //use: nodeId[phPin.id]
			for(var i=0; i<groups.length; i++){
				for(var j=0; j<groups[i].length; j++){
					nodeId[groups[i][j]] = i+1; //+1, to avoid id 0, which corresponds to gnd
				}
			}

			//set ground
			const phGroundId = that.phGround.id;
			const groundGroupIndex = nodeId[phGroundId] - 1;
			for(var j=0; j<groups[groundGroupIndex].length; j++){
				nodeId[groups[groundGroupIndex][j]] = 0;
			}

			return nodeId;
		}

		const spNodeId = getIdMap(this.graph); //just bare metal
		const allPotNodeId = getIdMap(this.graphAllPot); //includes inner connections
														 //used to exclude unused items from spice

		// each spice node must have a dc path to ground
		// exclude those that don't
		function connectedPhPins(ce){
			let connected = [];
			for(let phPin of ce.phPins){
				if(allPotNodeId[phPin.id] == 0){
					connected.push(phPin);
				}
			}
			return connected;
		}

		// to be passed to QT
		let info = {
			oscilloscope: {
				CH1: {
					connected: false
				},
				CH2: {
					connected: false
				},
				EXT: {
					connected: false
				}
			},
			multimeter: {
				ammeter: {connected: false},
				voltmeter: {connected: false}
			},
			formData: {},
			studentnumber: document.getElementById("studentnumber").value,
			spice: ''
		};

		let spice = "breadboard simulation\n";

		for(let ce of this.elements){
			if(ce === null){
				continue;
			}
			switch(ce.type){
				case 'resistor':
				case 'inductor':
				case 'capacitor':
					if(connectedPhPins(ce).length != 2){
						break;
					}
					spice += `${ce.spiceId()} ${spNodeId[ce.phPins[0].id]} ${spNodeId[ce.phPins[1].id]} ${ce.spiceValue()}`;
					if(ce.type != 'resistor') {
						spice += ' ic=0';
					}
					spice += "\n";
					info.formData[ce.spiceId()] = ce.formData;
					break;
				case 'oscilloscope':
					if(allPotNodeId[ce.ch1Core.id] == 0 && allPotNodeId[ce.ch1Shield.id] == 0){
						info.oscilloscope.CH1.nodeA = `V(${spNodeId[ce.ch1Core.id]})`
						info.oscilloscope.CH1.nodeB = `V(${spNodeId[ce.ch1Shield.id]})`; //TODO: check order
						info.oscilloscope.CH1.connected = true;
					}
					if(allPotNodeId[ce.ch2Core.id] == 0 && allPotNodeId[ce.ch2Shield.id] == 0){
						info.oscilloscope.CH2.nodeA = `V(${spNodeId[ce.ch2Core.id]})`
						info.oscilloscope.CH2.nodeB = `V(${spNodeId[ce.ch2Shield.id]})`;
						info.oscilloscope.CH2.connected = true;
					}
					if(allPotNodeId[ce.extCore.id] == 0 && allPotNodeId[ce.extShield.id] == 0){
						info.oscilloscope.EXT.nodeA = `V(${spNodeId[ce.extCore.id]})`
						info.oscilloscope.EXT.nodeB = `V(${spNodeId[ce.extShield.id]})`;
						info.oscilloscope.EXT.connected = true;
					}
					break;
				case 'multimeter':
					if(allPotNodeId[ce.phA.id] == 0 && allPotNodeId[ce.phCom.id] == 0){
						spice += `Vammeter ${spNodeId[ce.phCom.id]} ${spNodeId[ce.phA.id]}\n` //TODO: check order
						// todo: implement ammeter
						info.multimeter.ammeter.connected = true;
                    }
                    if(allPotNodeId[ce.phV.id] == 0 && allPotNodeId[ce.phCom.id] == 0){
						info.multimeter.voltmeter.nodeA = `V(${spNodeId[ce.phV.id]})`;
						info.multimeter.voltmeter.nodeB = `V(${spNodeId[ce.phCom.id]})`; //TODO: check order
						info.multimeter.voltmeter.connected = true;
					}
					break;
				case 'signal-generator':
					spice += `VIN ${spNodeId[ce.phPins[0].id]} ${spNodeId[ce.phPins[1].id]} __SIGNALGENERATOR__\n`;
					break;
			}
		}

		spice += "\n";
		info.spice = spice;

		backend.gotSpice(info);
	}
}

class Connection{ //between 2 Pins
	constructor(pinA, pinB, line){
		this.pinA = pinA;
		this.pinB = pinB;
		this.line = line;
	}

}

class Pin { // can be connected by lines on the GUI
	constructor(domElement, circuitElement, connection=null, type='wire', phPins = []){
		this.domElement = domElement;
		this.circuitElement = circuitElement;
		this.connection = connection;
		this.type = type;
		this.phPins = phPins; // corresponding Pin or Pins in case of coax
	}

	clicked(event){
		const circuit = this.circuitElement.circuit;
		if(circuit.pinSelectA === null){
			// if already connected, disconnect
			if(this.connection){
				circuit.disconnect(this);
			}else{
				// select pin
				circuit.pinSelectA = this;
				this.domElement.classList.add("selected");
			}
		}else if(circuit.pinSelectA.domElement == this.domElement){
			circuit.pinSelectA = null;
			this.domElement.classList.remove("selected");
		}else if(this.connection){
			// dont't allow 2 connections to same pin
		}else{
			circuit.connect(this);
		}
	}

	anchorPos(){
		let x, y;
		if(this.circuitElement.type == 'resistor'){
			if(this.domElement.classList.contains('left')){
				x=0; y=50;
			}else{
				x=100; y=50;
			}
		}

		if(this.circuitElement.type == 'inductor'){
			if(this.domElement.classList.contains('left')){
				x=0; y=57;
			}else{
				x=100; y=57;
			}
		}

		if(this.circuitElement.type == 'capacitor'){
			if(this.domElement.classList.contains('left')){
				x=45; y=100;
			}else{
				x=53; y=100;
			}
		}

		if(this.circuitElement.type == 'clips'){
			if(this.domElement.classList.contains('BLACK')){
				x=43; y=0;
			}else if(this.domElement.classList.contains('RED')){
				x=67; y=0;
			}else{
				x=52; y=100;
			}
		}

		if(this.circuitElement.type == 'tee'){
			if(this.domElement.classList.contains('OUT1')){
				x=5; y=40;
			}else if(this.domElement.classList.contains('OUT2')){
				x=95; y=40;
			}else{//IN
				x=50; y=100;
			}
		}

		switch(this.circuitElement.orientation){
			case compass.EAST:
				return {x: x+'%', y: y+'%'};
			case compass.SOUTH:
				return {x: (100-y)+'%', y: x+'%'};
			case compass.WEST:
				return {x: (100-x)+'%', y: (100-y)+'%'};
			case compass.NORTH:
				return {x: y+'%', y: (100-x)+'%'};
		}
	}
}

class PhPin { //Physical pins: connected by bare metal, on same potential in DC approx.
	constructor(circuitElement, name, pin){
		this.circuitElement = circuitElement;
		this.name = name; // which pin of the circuit element, eg. ground
		this.id = circuitElement.circuit.newPhPinId();

		// create node in graph
		this.circuitElement.circuit.graph.setNode(this.id, this);
		this.circuitElement.circuit.graphAllPot.setNode(this.id, this);
	}
}

class CircuitElement {
	constructor(id, circuit){
		this.orientation = compass.EAST;
		this.id = id;
		this.circuit = circuit;
		this.pins = [];
		this.phPins = [];
		this.isDraggable = false;
		this.rotatable = true;
		this.domElement = false;
		this.formData = {};
	}

	draw(){
		const canvas = document.getElementById("canvas");
		let domElement = document.createElement("div");
		let html = '<div class="overlay"><div class="pin left"></div><div class="pin right"></div></div><object data="' + this.url + '"></object>';
		domElement.classList.add('ce', this.type);
		domElement.id = this.type + this.id;
		domElement.innerHTML = html;
		this.domElement = domElement;
		canvas.appendChild(this.domElement);

		// make element draggable
		this.draggable = new PlainDraggable(domElement, {
			onMove: this.moved.bind(this)
		});

		// add right click menu
		this.addContextMenu();

		// add pin select listeners
		let left = new Pin(domElement.querySelector(".left"), this);
		let right = new Pin(domElement.querySelector(".right"), this);
		left.domElement.addEventListener("dblclick", left.clicked.bind(left));
		right.domElement.addEventListener("dblclick", right.clicked.bind(right));
		this.pins = [left, right];

		let phLeft = new PhPin(this, 'left');
		let phRight = new PhPin(this, 'right');
		left.phPins = [phLeft];
		right.phPins = [phRight];
		this.phPins = [phLeft, phRight];
	}

	// add edges between physical pins within the same circuit element
	addInnerEdges(){
		this.circuit.graphAllPot.setEdge(this.phPins[0].id, this.phPins[1].id);
	}

	humanId(){
		return null;
	}

	spiceId(){
		return null;
	}

	spiceValue(){
		if(this.formData.value === undefined){
			return;
		}

		const prefixMap = {
			'f': 'f', 'p': 'p', 'n': 'n',
			'μ': 'u', 'm': 'm', 'k': 'k',
			'M': 'meg', 'G': 'g'
		};

		let value = `__${this.spiceId()}_VALUE__`;
		//let value = parseFloat(this.formData.value);

		if(this.formData.suffix.length > 1){
			value += prefixMap[this.formData.suffix[0]];
		}

		return value;
	}

	addContextMenu(){
		this.menu = CtxMenu(this.domElement);
		let humanId = this.humanId();
		if(humanId){
			this.menu.addItem(humanId, noop);
		}
		if(this.rotatable){
			this.menu.addItem("Rotate &#8635;", this.rotate.bind(this, 1));
			this.menu.addItem("Rotate &#8634;", this.rotate.bind(this, 3));
		}
		this.menu.addItem("Remove", this.delete.bind(this));
	}

	moved(){
		// redraw wire lines
		for(var i = 0; i<this.pins.length; i++){
			let connection = this.pins[i].connection;
			if(connection){
				connection.line.position();
			}
		}
	}

	// rotate by n*90° clockwise
	static rotateMapping(direction, n){
		const map = {
			"left": "top", "right": "bottom",
			"bottom": "left", "top": "right",
			"auto": "auto"
		};

		for(var i=0; i<n; i++){
			direction = map[direction];
		}

		return direction;
	}

	rotate(n=1){
		this.orientation = (this.orientation + n) % 4;
		this.domElement.classList.remove('orientation0', 'orientation1', 'orientation2', 'orientation3');
		this.domElement.classList.add('orientation' + this.orientation);

		let endSocket;
		let startSocket;
		for(var i = 0; i<this.pins.length; i++){
			let connection = this.pins[i].connection;
			if(!connection){
				continue;
			}

			let anchorA =  LeaderLine.pointAnchor(connection.pinA.domElement, connection.pinA.anchorPos());
			let anchorB =  LeaderLine.pointAnchor(connection.pinB.domElement, connection.pinB.anchorPos());

			connection.line.setOptions({
				start: anchorA,
				end: anchorB,
				startSocket: CircuitElement.rotateMapping(connection.line.startSocket,n),
				endSocket: CircuitElement.rotateMapping(connection.line.endSocket,n)
			});
		}


	}

	socketOrientation(pin){
		return pin.domElement.classList.contains("left") ? "left" : "right";
	}

	delete(){
		delete this.menu;
		this.draggable.remove();
		delete this.draggable;
		for(var i = 0; i<this.pins.length; i++){
			this.circuit.disconnect(this.pins[i]);
		}
		for(const phPin of this.phPins){
			this.circuit.graph.removeNode(phPin.id);
			this.circuit.graphAllPot.removeNode(phPin.id);
		}
		this.circuit.elements[this.id] = null;
		this.domElement.remove();
		delete this.domElement;

		let vuelist = vue.$data.circuitElements;
		for(i = 0; i < vuelist.length; i++){
			if(this.id == vuelist[i].id){
				Vue.delete(vue.$data.circuitElements, i);
			}
		}

		delete this;
	}


}

class Resistor extends CircuitElement {
	constructor(id, circuit){
		super(id, circuit);
		this.url = 'svg/resistor_220.svg';
		this.type = 'resistor';

		this.isDraggable = true;

		this.draw();
		this.addInnerEdges();
	}

	humanId(){
		return 'R<sub>' + this.id + '</sub>';
	}

	spiceId(){
		return 'R' + this.id;
	}
}

class Inductor extends CircuitElement {
	constructor(id, circuit){
		super(id, circuit);
		this.url = 'svg/inductor_leg.svg';
		this.type = 'inductor';

		this.isDraggable = true;

		this.draw();
		this.addInnerEdges();
	}

	humanId(){
		return 'L<sub>' + this.id + '</sub>';
	}

	spiceId(){
		return 'L' + this.id;
	}
}

class Capacitor extends CircuitElement {
	constructor(id, circuit){
		super(id, circuit);
		this.url = 'svg/ceramic_capacitor_blue_leg.svg';
		this.type = 'capacitor';

		this.isDraggable = true;

		this.draw();
		this.addInnerEdges();
	}

	humanId(){
		return 'C<sub>' + this.id + '</sub>';
	}

	spiceId(){
		return 'C' + this.id;
	}

	socketOrientation(pin){
		return 'bottom';
	}
}

class BreadboardGroup extends CircuitElement {
	constructor(id, circuit, pinIds){
		super(id, circuit);
		this.type = 'breadboard';

		this.draw(pinIds);
		this.addInnerEdges();
	}

	draw(pinIds){
		for(var i = 0; i<pinIds.length; i++){
			// add pin select listeners
			let pin = new Pin(document.getElementById(pinIds[i]), this);
			let phPin = new PhPin(this, 'bb');
			pin.domElement.addEventListener("dblclick", pin.clicked.bind(pin));
			pin.phPins = [phPin];
			this.pins.push(pin);
			this.phPins.push(phPin);
		}
	}

	addInnerEdges(){
		//connect adjacent pins
		for(let i=1; i<this.phPins.length; i++){
			this.circuit.graph.setEdge(this.phPins[i-1].id, this.phPins[i].id);
			this.circuit.graphAllPot.setEdge(this.phPins[i-1].id, this.phPins[i].id);
		}
	}

	socketOrientation(){
		return 'auto';
	}
}

class Oscilloscope extends CircuitElement {
	constructor(id, circuit){
		super(id, circuit);
		this.url = 'svg/oscilloscope.svg';
		this.type = 'oscilloscope';

		this.isDraggable = false;

		this.draw();
	}

	socketOrientation(pin){
		return 'auto';
	}

	draw(){
		const canvas = document.getElementById("canvas");
		let domElement = document.createElement("div");
		let html = '<div class="overlay"><div class="pin ch1 bnc"></div><div class="pin ch2 bnc"></div><div class="pin ext bnc"></div></div><object data="' + this.url + '"></object>';
		domElement.classList.add('ce', this.type);
		domElement.id = this.type + this.id;
		domElement.innerHTML = html;
		domElement.style.transform = 'translate(14px, 110px)';
		this.domElement = domElement;
		canvas.appendChild(this.domElement);

		// make element draggable
		this.draggable = new PlainDraggable(domElement, {
			onMove: this.moved.bind(this)
		});

		//create physical pins
		let ch1Core   = new PhPin(this, 'CH1 core');
		let ch1Shield = new PhPin(this, 'CH1 shield');
		let ch2Core   = new PhPin(this, 'CH2 core');
		let ch2Shield = new PhPin(this, 'CH2 shield');
		let extCore   = new PhPin(this, 'EXT core');
		let extShield = new PhPin(this, 'EXT shield');
		this.phPins   = [
			ch1Core, ch1Shield,
			ch2Core, ch2Shield,
			extCore, extShield
		];

		[this.ch1Core, this.ch1Shield, this.ch2Core, this.ch2Shield, this.extCore, this.extShield] = 
		[ch1Core,      ch1Shield,      ch2Core,      ch2Shield,      extCore,      extShield ];

		// add pin select listeners
		let ch1 = new Pin(domElement.querySelector(".ch1"), this, null, 'bnc', [ch1Core, ch1Shield]);
		ch1.domElement.addEventListener("dblclick", ch1.clicked.bind(ch1));
		let ch2 = new Pin(domElement.querySelector(".ch2"), this, null, 'bnc', [ch2Core, ch2Shield]);
		ch2.domElement.addEventListener("dblclick", ch2.clicked.bind(ch2));
		let ext = new Pin(domElement.querySelector(".ext"), this, null, 'bnc', [extCore, extShield]);
		ext.domElement.addEventListener("dblclick", ext.clicked.bind(ext));
		
		this.pins = [ch1, ch2, ext];

	}
}

class SignalGenerator extends CircuitElement {
	constructor(id, circuit){
		super(id, circuit);
		this.url = 'svg/signal_generator.svg';
		this.type = 'signal-generator';

		this.isDraggable = false;

		this.draw();
	}

	socketOrientation(pin){
		return 'auto';
	}

	draw(){
		const canvas = document.getElementById("canvas");
		let domElement = document.createElement("div");
		let html = '<div class="overlay"><div class="pin out bnc"></div></div><object data="' + this.url + '"></object>';
		domElement.classList.add('ce', this.type);
		domElement.id = this.type + this.id;
		domElement.innerHTML = html;
		domElement.style.transform = 'translate(586px, 109px)';
		this.domElement = domElement;
		canvas.appendChild(this.domElement);

		// make element draggable
		this.draggable = new PlainDraggable(domElement, {
			onMove: this.moved.bind(this)
		});

		//create physical pins
		let outCore   = new PhPin(this, 'core');
		let outShield = new PhPin(this, 'GND');
		this.circuit.phGround = outShield;
		this.phPins   = [outCore, outShield];
		this.circuit.graphAllPot.setEdge(outCore.id, outShield.id);

		// add pin select listeners
		//TODO: add counter, CMOS
		let out = new Pin(domElement.querySelector(".out"), this, null, 'bnc', [outCore, outShield]);
		out.domElement.addEventListener("dblclick", out.clicked.bind(out));

		this.pins = [out];
	}
}

class Multimeter extends CircuitElement {
	constructor(id, circuit){
		super(id, circuit);
		this.url = 'svg/multimeter.svg';
		this.type = 'multimeter';

		this.isDraggable = false;

		this.draw();
	}

	socketOrientation(pin){
		return 'auto';
	}

	draw(){
		const canvas = document.getElementById("canvas");
		let domElement = document.createElement("div");
		let html = '<div class="overlay"><div class="pin A"></div><div class="pin COM"></div><div class="pin V"></div></div><object data="' + this.url + '"></object>';
		domElement.classList.add('ce', this.type);
		domElement.id = this.type + this.id;
		domElement.innerHTML = html;
		domElement.style.transform = 'translate(508px, -156.7px)';
		this.domElement = domElement;
		canvas.appendChild(this.domElement);

		// make element draggable
		this.draggable = new PlainDraggable(domElement, {
			onMove: this.moved.bind(this)
		});

		//create physical pins
		let phA   = new PhPin(this, 'probe');
		let phCom = new PhPin(this, 'probe');
		let phV   = new PhPin(this, 'probe');
		this.phPins   = [phA, phCom, phV];
		[this.phA, this.phCom, this.phV] = [phA, phCom, phV];

		this.circuit.graphAllPot.setEdge(phA.id, phCom.id);

		// add pin select listeners
		let A = new Pin(domElement.querySelector(".A"), this);
		A.domElement.addEventListener("dblclick", A.clicked.bind(A));
		let com = new Pin(domElement.querySelector(".COM"), this);
		com.domElement.addEventListener("dblclick", com.clicked.bind(com));
		let V = new Pin(domElement.querySelector(".V"), this);
		V.domElement.addEventListener("dblclick", V.clicked.bind(V));

		A.phPins = [phA];
		com.phPins = [phCom];
		V.phPins = [phV];

		this.pins = [A, com, V];
	}
}

class Tee extends CircuitElement {
	constructor(id, circuit){
		super(id, circuit);
		this.url = 'svg/tee.svg';
		this.type = 'tee';

		this.isDraggable = false;
		this.isRotatable = true;

		this.draw();
	}

	socketOrientation(pin){
		if(pin.domElement.classList.contains("OUT1")){
			return 'left';
		}else if(pin.domElement.classList.contains("OUT2")){
			return 'right';
		}else{
			return 'bottom';
		}
	}

	draw(){
		const canvas = document.getElementById("canvas");
		let domElement = document.createElement("div");
		let html = '<div class="overlay"><div class="pin IN bnc"></div><div class="pin OUT1 bnc"></div><div class="pin OUT2 bnc"></div></div><object data="' + this.url + '"></object>';
		domElement.classList.add('ce', this.type);
		domElement.id = this.type + this.id;
		domElement.innerHTML = html;
		domElement.style.transform = 'translate(240px, 32px)';
		this.domElement = domElement;
		canvas.appendChild(this.domElement);

		// make element draggable
		this.draggable = new PlainDraggable(domElement, {
			onMove: this.moved.bind(this)
		});

		// add right click menu
		this.addContextMenu();

		//create physical pins
		let out1Core   = new PhPin(this, 'core');
		let out1Shield = new PhPin(this, 'shield');
		let out2Core   = new PhPin(this, 'core');
		let out2Shield = new PhPin(this, 'shield');
		let inpCore    = new PhPin(this, 'core');
		let inpShield  = new PhPin(this, 'shield');
		this.phPins   = [
			out1Core, out1Shield,
			out2Core, out2Shield,
			inpCore, inpShield
		];

		// add inner connections to graph
		this.circuit.graph.setEdge(out1Core.id, out2Core.id);
		this.circuit.graph.setEdge(out1Core.id, inpCore.id);
		this.circuit.graph.setEdge(out1Shield.id, out2Shield.id);
		this.circuit.graph.setEdge(out1Shield.id, inpShield.id);

		this.circuit.graphAllPot.setEdge(out1Core.id, out2Core.id);
		this.circuit.graphAllPot.setEdge(out1Core.id, inpCore.id);
		this.circuit.graphAllPot.setEdge(out1Shield.id, out2Shield.id);
		this.circuit.graphAllPot.setEdge(out1Shield.id, inpShield.id);

		// add pin select listeners
		let inp = new Pin(domElement.querySelector(".IN"), this, null, 'bnc', [inpCore, inpShield]);
		inp.domElement.addEventListener("dblclick", inp.clicked.bind(inp));
		let out1 = new Pin(domElement.querySelector(".OUT1"), this, null, 'bnc', [out1Core, out1Shield]);
		out1.domElement.addEventListener("dblclick", out1.clicked.bind(out1));
		let out2 = new Pin(domElement.querySelector(".OUT2"), this, null, 'bnc', [out2Core, out2Shield]);
		out2.domElement.addEventListener("dblclick", out2.clicked.bind(out2));
		
		this.pins = [inp, out1, out2];
	}


}

class Clips extends CircuitElement {
	constructor(id, circuit){
		super(id, circuit);
		this.url = 'svg/croc.svg';
		this.type = 'clips';

		this.isDraggable = true;
		this.rotatable = true;

		this.draw();
	}

	draw(){
		const canvas = document.getElementById("canvas");
		let domElement = document.createElement("div");
		let html = '<div class="overlay"><div class="pin BLACK"></div><div class="pin RED"></div><div class="pin BNC bnc"></div></div><object data="' + this.url + '"></object>';
		domElement.classList.add('ce', this.type);
		domElement.id = this.type + this.id;
		domElement.innerHTML = html;
		domElement.style.transform = 'translate(654px, -117px)';
		this.domElement = domElement;
		canvas.appendChild(this.domElement);

		// make element draggable
		this.draggable = new PlainDraggable(domElement, {
			onMove: this.moved.bind(this)
		});

		// add right click menu
		this.addContextMenu();

		//create physical pins
		let bncCore   = new PhPin(this, 'core');
		let bncShield = new PhPin(this, 'shield');
		let phBlack   = new PhPin(this, 'croc');
		let phRed     = new PhPin(this, 'croc');
		this.phPins   = [
			bncCore, bncShield,
			phBlack, phRed
		];

		// add inner connections to graph
		this.circuit.graph.setEdge(bncCore.id, phRed.id);
		this.circuit.graph.setEdge(bncShield.id, phBlack.id);
		this.circuit.graphAllPot.setEdge(bncCore.id, phRed.id);
		this.circuit.graphAllPot.setEdge(bncShield.id, phBlack.id);

		// add pin select listeners
		let black = new Pin(domElement.querySelector(".BLACK"), this,);
		black.domElement.addEventListener("dblclick", black.clicked.bind(black));
		black.phPins = [phBlack];
		let red = new Pin(domElement.querySelector(".RED"), this);
		red.domElement.addEventListener("dblclick", red.clicked.bind(red));
		red.phPins = [phRed];
		let bnc = new Pin(domElement.querySelector(".BNC"), this, null, 'bnc', [bncCore, bncShield]);
		bnc.domElement.addEventListener("dblclick", bnc.clicked.bind(bnc));
		
		this.pins = [black, red, bnc];

	}

	socketOrientation(pin){
		if(pin.domElement.classList.contains("BNC")){
			return 'bottom';
		}else{
			return 'top';
		}
	}
}


function loadSVG(){
	PlainDraggable.draggableCursor = 'pointer';
	PlainDraggable.draggingCursor = ['grabbing', 'move'];

	circuit = new Circuit;
	backend.getSpice.connect(circuit.getSpice.bind(circuit));
	backend.toggleEnabled.connect(function(){
		let block = document.getElementById("block");
		if(backend.webenabled){
			block.style.display = 'block';
		}else{
			block.style.display = 'none';
		}
	});

	Vue.component('resistor', {
		data: function () {
			return {
				value: 3.5,
				suffix: 'kΩ',
				tolerance: 5
			}
		},
		created: function(){
			this.ce.formData = this.$data;
		},
		props: ["ce"],
		template: '#resistor-template'
	});

	Vue.component('inductor', {
		data: function () {
			return {
				value: 1.0,
				suffix: 'μH',
				tolerance: 5
			}
		},
		created: function(){
			this.ce.formData = this.$data;
		},
		props: ["ce"],
		template: '#inductor-template'
	});

	Vue.component('capacitor', {
		data: function () {
			return {
				value: 1.0,
				suffix: 'nF',
				tolerance: 5
			}
		},
		created: function(){
			this.ce.formData = this.$data;
		},
		props: ["ce"],
		template: '#capacitor-template'
	});

	vue = new Vue({
		el: '#control',
		data: {
			circuitElements: [],
			studentnumber: ''
		},
		methods: {
			newElement: function(type){
				let element = circuit.newElement(type);
				this.circuitElements.push(element);
			},
			updateSpice: function(){
				circuit.getSpice();
			}
		}
	});
}

var backend;

new QWebChannel(qt.webChannelTransport, function(channel) {
	backend = channel.objects.backend;
	backend.loadBbSvg(function(bbsvg) {
		let svg = document.getElementById("breadboardSVG");
		svg.innerHTML = bbsvg;
		loadSVG();
	});
});
