# _qTAtrainer
version$ = "v 1.3"
#	v 1.9 = _PENTAtrainer1 v 1.9.3.9
# Renamed 25 November, 2019
# Last update: 9 September, 2024
# Written by:   Yi Xu (yi.xu@ucl.ac.uk) and Santitham Prom-on 
# (Santitham Prom-on <santitham@cpe.kmutt.ac.th>); all rights reserved.
# Report any bugs to Yi Xu

#	This script is for automatic extraction of pitch target parameters. A pitch target is an ideal 
#	f0 trajectory associated with a segmental unit, which is defined by three parameters: slope, 
#	height and strength. The target notion is the core of the PENTA model (Parallel Encoding and 
#	Target Approximation, cf. Xu, 2005). The current script is based on the qTA implementation of 
#	PENTA (Prom-on et al., 2009).

#	In qTA, a target is defined by a linear equation f0 = mt + b, where f0 is the surface f0, m is 
#	the slope of the target and b is the height of the target defined as the intercept of the target 
#	offset with the y-axis. The surface f0 is the outcome of sequential asymptotic approximation of 
#	successive pitch targets based on a critically damped 3rd-order linear system.

#	The extraction of target parameters in this script is done by analysis-by-synthesis. For each 
#	target interval, the surface f0 contour is generated based on qTA using all combinations of the 
#	three parameters within the search range at a certain step size, and the difference between the 
#	synthesized and original contours is computed in terms of sum of squared errors (SSE). 
#	The parameter combination with the least SSE is chosen as the target of the interval.

#	The target intervals are defined by user by marking and naming it with a label in the top tier 
#	of the TextGrid. Intervals with no labels are ignored by the script.

#	The target search ranges can be restricted by user in a number of ways:

# 1.	In the startup window, users can change the global search ranges defined by the maximum and 
#	minimum parameter values.
# 2.	For each sound file, empty intervals in the target tier (2) are given full search ranges 
#	defined by the maximum and minimum parameter values. 
# 3.	Tier 2 intervals labeled as H, M, L, h, m or l are given a fixed slope at 0. 
# 4.	Tier 2 intervals labeled as R or r are searched only for positive slopes. 
# 5.	Tier 2 intervals labeled as F or f are searched only for negative slopes

#	In addition to target extraction, the script also performs all the f0 analysis inherited from 
#	ProsodyPro (http://www.phon.ucl.ac.uk/home/yi/ProsodyPro/, as explained below.

# SYNOPSIS: 
# 1) Automatically open each .wav file in a folder, manually label intervals and rectify 
# vocal pulse markings; 
# 2) Save automatically trimmed (smoothed) f0 contours; 
# 3) Save time-normalized f0; 
# 4) Save time-normalized f0 with actual time; 
# 5) Save manually rectified, trimmed f0 as PitchTier objects which can replace the pitch 
# tier in Manipulation objects; 
# 6) Save sampled f0; 
# 7) Save f0 velocity; 
# 8) Save maxf0, minf0, excursionsize, mean f0, intensity, duration, peak velocity, final velocity 
#	and final f0 of labeled intervals;
# 9) Save results into ensemble files.

# INSTRUCTIONS:

# 1. Put all the (or a group of) sound files to be analyzed in a single folder together with 
# this script, and launch Praat;

# 2. Select Open Praat Script... from the top menu;

# 3. Locate this script in the dialogue window and select it;

# 4. When the script window opens in Praat, select Run from the Run menu (or type the key 
# shortcut command-r or control-r);

# 5. When a dialogue window opens, check or uncheck the boxes according to your analysis needs, 
# and set appropriate values in the text fields or simply use the default values.

# 6. Click OK and three windows will appear. The first big window displays the waveform together 
# with vocal cycle marks (vertical lines) generated by Praat. This is where you can manually 
# make sure the F0 tracking is accurate by adding marks to those vocal cycles missed by Praat 
# and deleting any redundant marks. But you need to do so only for the named intervals, as 
# explained next.

# 7. The second big window displays the waveform and spectrogram of the current sound together 
# with optional pitch track and formant tracks in the spectrogram panel, and vocal pulse marks 
# in the waveform panel. (These tracks and marks cannot be manually changed. So you can hide 
# them to reduce processing time by using the corresponding menu.) At the bottom of this window 
# are two TextGrids, where you can insert interval boundaries (upper grid) and point labels 
# (lower grid). Most importantly, for any interval that you want to have analysis results saved, 
# you need to type a name. Make the name as simple as possible to save time, e.g., a, b, or 1, 2.

# 8. The third window (qTAtrainer) displays pitch targets (green dashed straight lines) and 
# synthesized f0 (red solid curve) against the original f0 (blue dashed curve). The thickness of a 
# target line represents its strength. The grey vertical lines indicate interval boundaries. When 
# there are no labeled intervals, only the original f0 is displayed.

# 9. After labeling the intervals, press "Replot" on the left side of the window and you will see 
# both synthesized and original f0 contours.

# 10. The qTAtrainer window allows you to inspect the f0 contours in various ways: zooming in 
# and out, scrolling left and right, and playing part or the whole of the original or resynthesized 
# signal. The window also allows you to move to the next or previous sound file.

# 11. When you click "Next" or "Previous" in the qTAtrainer window, the TextGrid and PointProcess
#  windows will be refreshed, displaying the spectrogram, waveform and vocal cycle marks of the next
# sound. You can repeat this process until all the sounds in the folder are processed. Or you can 
# finish any time by clicking "Exit".

# 12. To modify the automatically learned parameters, or use your own parameters, change them in the
# TextGrid window, after which you can press the Replot button in the qTAtrainer window to see and
# hear the newly synthesized f0 contours.

# 13. You can make qTAtrainer skip a voiceless region by assigning it a blank interval. Any 
# blank interval with duration < minimum_pause_duration will be treated as a syllable-initial 
# voiceless consonant. Note, however, the skipping of voiceless consonants is not obligatory for parameter extraction.

# 14. The following files will be saved as each sound file is processed:  

# 	.f0 -- trimmed f0 with real time.

# 	.rawf0 -- raw with real time computed directly from the pulse markings.

# 	.normtimef0 -- time-normalized f0. The f0 in each interval is divided 
# 	into the same number of points (default = 10). Thus points 1-10 belong to interval 1, and 
# 	points 11-20 belong to interval 2, etc.

# 	.actutimenormf0 -- time-normalized f0 with each interval divided into the same number
# 	of points (default = 10). But the time scale is the original, except
# 	that the onset time of interval 1 is set to 0, unless the "Set initial time to 0" box in 
# 	the initial dialogue window is unchecked.  

# 	.samplef0 -- f0 values at fixed time intervals specified by "f0 sample rate".    

# 	.f0velocity -- velocity profile (instantaneous rates of F0 change) of f0 contour in 
# 	semitone/s at fixed time intervals specified by "f0 sample rate".

# 	.means -- containing various measurements, including maxf0, minf0, excursion size,  
# 	mean intensity, duration, max_velocity, final_velocity. Final velocity is taken at a point
#   earlier than the interval offset by time specified by "Final offset" in the 
#   startup window. It is an indicator of the slope of the underlying target of the interval.
#	Final f0 is is an indicator of the height of the underlying target, taken also at a point
#   specified by "Final offset" in the startup dialogue window.

# 15. The values in these files correspond only to the named intervals, i.e., those with 
# text names. No values are saved for the blank intervals.

# 16. The .f0, .normtimef0, and .actutimenormf0 files can be opened by a graphing program 
# such as Excel. The left-hand column is time and the right-hand column f0.

# 17. The .means file contains values of maxf0, minf0, excursionsize, meanf0, mean intensity, duration and 
# peak velocity (if "Get sample f0 box" has been checked when processing individual files) of 
# all named intervals.

# 18. After the analysis of all the individual sound files are done, you can gather the analysis 
# results into a number of ensemble files by running the script again and checking the "Get 
# ensemble results" button in the initial dialogue window. The following ensemble files are saved:

# 		1)	normf0.txt
# 		2)	normactutime.txt
# 		3)	samplef0.txt
# 		4)	f0velocity.txt
# 		5)	maxf0.txt
# 		6)	minf0.txt
# 		7)	excursionsize.txt
# 		8)	meanf0.txt
# 		9)	duration.txt
# 		10)	maxvelocity.txt
# 		11)	finalvelocity.txt
# 		12)	finalf0.txt
# 		13)	meanintensity.txt

# 19. Note that you can generate an ensemble file only if you have saved at least one  
# corresponding individual analysis file described earlier.


form qTAtrainer
	integer Input_File_No 1
	word TextGrid_extension label
	optionmenu Task: 1
		option Interactive target view
		option Process all sounds without pause
		option Get emsemble files
		option Synthesis with arbitrary targets
	comment F0 analysis options:
		integer left_F0_range_(Hz) 30
		integer right_F0_range_(Hz) 400
		integer Npoints_per_interval 10
		integer F0_sample_rate_(Hz) 100
		integer Perturbation_length_(ms) 0
		real Final_offset -0.03
		real Smoothing_window_width_(s) 0.07
		boolean Set_initial_time_for_normf0_to_0 1
	comment Target extraction options:
		real left_Target_slope_range_(st/s) -100
		real right_Target_slope_range_(st/s) 100
		real left_Target_height_range_(st) -20
		real right_Target_height_range_(st) 20
		real left_Strength_range 1
		real right_Strength_range 80
		real minimum_pause_duration_(ms) 150
		boolean Save_synthetic_sounds 0
		word qTA_synthesis_folder qTA_synthesis
		word Silence_marker_(optional) sil
		boolean Ignore_target_labels_in_Tier_2 1
		boolean Use_final_velocity_to_constrain_slope 0
		boolean Fix_strength_to_minimum_strength 0
endform

if task = 3
	printline 'newline$' 	Collecting data from all individual files. Please wait patiently...'newline$'
endif

npoints = npoints_per_interval
minf0 = left_F0_range
maxf0 = right_F0_range
min_target_slope = left_Target_slope_range
max_target_slope = right_Target_slope_range
min_target_height = left_Target_height_range
max_target_height = right_Target_height_range
min_strength = left_Strength_range
max_strength = right_Strength_range
fixed_strength = fix_strength_to_minimum_strength
if (fixed_strength<1)
	assert min_strength < max_strength
endif
assert min_target_slope < max_target_slope 
assert min_target_height < max_target_height
found_interval = 0
clicked_next = 0

b_Location = 2

directory$ = "./"
Create Strings as file list... list 'directory$'*.wav
numberOfFiles = Get number of strings
if !numberOfFiles
	Create Strings as file list... list 'directory$'*.WAV
	numberOfFiles = Get number of strings
endif
if !numberOfFiles
	exit There are no sound files in the folder!
else
	Write to raw text file... 'directory$'FileList.txt
	more_file = 1
endif
hasmeanstitle = 0
hasnormf0 = 0
hasnormactutime = 0
hassamplef0 = 0
hasf0velocity = 0
hasqTAf0 = 0
hasnewduration = 0
number = input_File_No
tier3$ = ""
for current_file from input_File_No to numberOfFiles
	select Strings list
	fileName$ = Get string... current_file
	name$ = fileName$ - ".wav" - ".WAV"
	if task == 3
		if fileReadable(directory$+name$+".normtimef0")
			call All_means 'name$'
			call All_normf0 'name$'
		endif
		if fileReadable(directory$+name$+".samplef0")
			call All_samplef0 'name$'
			call All_f0velocity 'name$'
		endif
	else
		call Labeling 'fileName$'
	endif
endfor

if task == 3
	echo Ensemble files saved:
	printline 1)	targets.txt
	printline 2)	normtimef0.txt
	printline 3)	normactutime.txt
	printline 4)	maxf0.txt
	printline 5)	minf0.txt
	printline 6)	excursionsize.txt
	printline 7)	meanf0.txt
	printline 8)	duration.txt
	printline 9)	maxvelocity.txt
	printline 10)	finalvelocity.txt
	printline 11)	finalf0.txt
	printline 12)	meanintensity.txt
	printline 13)	samplef0.txt
	printline 14)	f0velocity.txt
endif

procedure Labeling file_name$ file_extension$
	Read from file... 'directory$''file_name$'
	name$ = file_name$ - ".wav" - ".WAV"
	if fileReadable ("'directory$''name$'.target")
		Read from file... 'directory$''name$'.target
		nTiers = Get number of tiers
		if nTiers > 2
			tier3$ = Get tier name... 3
		endif
		if nTiers < 4 or (nTiers > 2 and tier3$ != "slope")
			nintervals = Get number of intervals... 1
			Duplicate tier... 1 2 target
			Replace interval text... 2 0 nintervals . "" Regular Expressions
			Duplicate tier... 2 3 slope
			Duplicate tier... 2 4 height
			Duplicate tier... 2 5 strength
			Duplicate tier... 2 6 duration
		endif
		interval = 0
	elsif fileReadable ("'directory$''name$'.TextGrid") or fileReadable ("'directory$''name$'.'TextGrid_extension$'")
		if fileReadable ("'directory$''name$'.TextGrid")
			Read from file... 'directory$''name$'.TextGrid
		endif
		if fileReadable ("'directory$''name$'.'TextGrid_extension$'")
			Read from file... 'directory$''name$'.'TextGrid_extension$'
		endif
########################################################
		select TextGrid 'name$'
		nTiers = Get number of tiers
		if nTiers > 2
			tier3$ = Get tier name... 3
		else
			tier3$ = ""
		endif
		if nTiers < 4 or (nTiers > 2 and tier3$ != "slope")
			nintervals = Get number of intervals... 1
			Duplicate tier... 1 2 target
			Replace interval text... 2 0 nintervals . "" Regular Expressions
			Duplicate tier... 2 3 slope
			Duplicate tier... 2 4 height
			Duplicate tier... 2 5 strength
			Duplicate tier... 2 6 duration
;Insert boundary: 2, 0.5
		endif
		interval = 0
#########################################################
	else
		To TextGrid... "interval target slope height strength duration comment" comment
	endif
	if task <> 2
		plus Sound 'name$'
		Edit
	endif
	
	pulsefile$ = name$+".pulse"
	if fileReadable (pulsefile$)
		Read from file... 'directory$''name$'.pulse
	else
		select Sound 'name$'
		To PointProcess (periodic, cc)... minf0 maxf0
		Rename... 'name$'
	endif
	if task <> 2
		plus Sound 'name$'
		Edit
	endif
	
	replot = 1
	replotting = 0
	call Save 'directory$' 'name$'
	select Sound 'name$'
	Remove
endproc

procedure Save directory$ name$
	select PointProcess 'name$'
	npulses = Get number of points
	for n from 2 to npulses
		time1 = Get time from index... n-1
		time2 = Get time from index... n
		if time2 = time1
			Remove point... n
		endif
	endfor
	Write to short text file... 'directory$''name$'.pulse
	maxperiod = 1/minf0
	To PitchTier... maxperiod
	Down to TableOfReal... Hertz
	Write to headerless spreadsheet file... 'directory$''name$'.rawf0
	Remove
	select PitchTier 'name$'
	call Trimf0
	Write to short text file... 'directory$''name$'.PitchTier
	Down to TableOfReal... Hertz
	Write to headerless spreadsheet file... 'directory$''name$'.f0
	Remove

	call Sampling
	if found_interval
		call Differentiation
		call Smooth "velocity" "smooth_velocity" 0.05 * f0_sample_rate
		select TableOfReal samplef0
		Write to headerless spreadsheet file... 'directory$''name$'.samplef0
		Remove
		select TableOfReal velocity
		Write to headerless spreadsheet file... 'directory$''name$'.f0velocity
	endif

	select TextGrid 'name$'
	Write to short text file... 'directory$''name$'.target

	if found_interval ; or found_interval and replot
		if replot or task = 2 or clicked_next
			call Normalization smoothf0 normf0 normactuf0
			select TableOfReal normf0
			Write to headerless spreadsheet file... 'directory$''name$'.normtimef0
			select TableOfReal normactuf0
			Write to headerless spreadsheet file... 'directory$''name$'.actutimenormf0
			plus TableOfReal normf0
			plus TableOfReal normactuf0
			Remove
		endif

		call Means
		select PitchTier semitonef0
		Write to short text file... 'directory$''name$'.PitchTier_semitone
		select PitchTier smooth_velocity
		Remove

		select TableOfReal means
		Write to headerless spreadsheet file... 'directory$''name$'.means
		plus Intensity 'name$'
		plus PitchTier samplef0
		plus PitchTier smoothf0
		plus PitchTier velocity
		plus PitchTier smoothvelocity
		plus TableOfReal smoothf0
		plus TableOfReal velocity
		plus TableOfReal smoothvelocity
		Remove
	endif
	
	if task = 1 and replot = 1
		if !found_interval
			select Sound 'name$'
			tInitial = Get start time
			tFinal = Get end time
		endif
		plot_width = tFinal - tInitial
		call plot_target tInitial tFinal
		select PitchTier 'name$'
		plus PitchTier semitonef0
		Remove
##################################
		call RemoveObject Pitch fittedf0_st
		call RemoveObject PitchTier fittedf0_st
		call RemoveObject Manipulation 'name$'
		call RemoveObject Sound 'name$'_qTA
##################################
	elsif task = 2
		select TextGrid 'name$'
		plus PointProcess 'name$'
		plus PitchTier 'name$'
		if found_interval
			plus PitchTier semitonef0
			plus PitchTier fittedf0_st
			plus Manipulation 'name$'
		else
			plus PitchTier samplef0
			plus TableOfReal samplef0
		endif
		Remove
	elsif task = 4 and replot = 1
		plot_width = tFinal - tInitial
		call plot_target tInitial tFinal
	endif	
endproc

procedure Trimf0
	maxbump = 0.01
	maxedge = 0.0
	maxgap = 0.033
	n = Get number of points
	
	first = Get value at index... 1
	second = Get value at index... 2
	penult = Get value at index... n-1
	last = Get value at index... n
	tfirst = Get time from index... 1
	tlast = Get time from index... n
	for k from 1 to 3
		call Trim
	endfor
	Remove point... 1
	Add point... tfirst second + (first-second) / 1000
	Remove point... n
	Add point... tlast penult + (last-penult) / 1000
endproc

procedure Trim
	for i from 2 to n-1
		tleft = Get time from index... i-1
		tmid = Get time from index... i
		tright = Get time from index... i+1
		gap1 = tmid - tleft
		gap2 = tright - tmid
		left = Get value at index... i-1
		mid = Get value at index... i
		right = Get value at index... i+1
		diff1 = mid - left
		diff2 = mid - right
		if diff1 > maxbump and diff2 > maxedge and gap1 < maxgap and gap2 < maxgap
		... or diff2 > maxbump and diff1 > maxedge and gap1 < maxgap and gap2 < maxgap
			Remove point... i
			Add point... tmid left+(tmid-tleft)/(tright-tleft)*(right-left)
		endif
		if diff1 > maxbump and gap2 >= maxgap
			Remove point... i
			Add point... tmid left + maxbump
		endif
		if diff2 > maxbump and gap1 >= maxgap
			Remove point... i
			Add point... tmid right + maxbump
		endif

		diff1 = left - mid
		diff2 = right - mid
		if diff1 > maxbump and diff2 > maxedge and gap1 < maxgap and gap2 < maxgap
		... or diff2 > maxbump and diff1 > maxedge and gap1 < maxgap and gap2 < maxgap
			Remove point... i
			Add point... tmid left+(tmid-tleft)/(tright-tleft)*(right-left)
		endif
		if diff1 > maxbump and gap2 >= maxgap
			Remove point... i
			Add point... tmid left - maxbump
		endif
		if diff2 > maxbump and gap1 >= maxgap
			Remove point... i
			Add point... tmid right - maxbump
		endif
	endfor
endproc

procedure Normalization pitchtier$ tableOfReal1$ tableOfReal2$
	if smoothing_window_width > 0
		select PitchTier 'pitchtier$'
	else
		select PitchTier 'name$'
	endif
	lasttime = Get finishing time
	Create PitchTier... normactutimef0 0 lasttime
	Create TableOfReal... 'tableOfReal1$' 1 2
	Set column label (index)... 1 NormalizedTime
	Set column label (index)... 2 F0
	Create TableOfReal... 'tableOfReal2$' 1 2
	Set column label (index)... 1 ActualTime
	Set column label (index)... 2 F0
	
	select TextGrid 'name$'
	nintervals = Get number of intervals... 1
	interval = 0
	found_interval = 0
	nrows = 0
	for m from 1 to nintervals
		select TextGrid 'name$'
		label$ = Get label of interval... 1 m
		start = Get starting point... 1 m
		end = Get end point... 1 m
		interval_dur = end - start
		if not label$ = "" and label$ != "'Silence_marker$'" and interval_dur >= 0.001
			if smoothing_window_width > 0
				select PitchTier 'pitchtier$'
			else
				select PitchTier 'name$'
			endif
			index1 = Get high index from time... start
			index2 = Get low index from time... end
			if found_interval = 0
				found_interval = 1
				if smoothing_window_width > 0
					select PitchTier 'pitchtier$'
				else
					select PitchTier 'name$'
				endif
				firstime = start
			endif
			call Normalize 'pitchtier$' 'tableOfReal1$' 'tableOfReal2$'
			interval = interval + 1
		endif
	endfor
	select TableOfReal 'tableOfReal1$'
	if nrows > 1
		Remove row (index)... nrows + 1
	endif
	select TableOfReal 'tableOfReal2$'
	if nrows > 1
		Remove row (index)... nrows + 1
	endif
	select PitchTier normactutimef0
	Remove
endproc

procedure Normalize pitchtier$ tableOfReal1$ tableOfReal2$
	duration = end - start

	for i from index1 to index2
		select PitchTier 'pitchtier$'
		time = Get time from index... i
		f0 = Get value at index... i
		select PitchTier normactutimef0
		Add point... (time-start)/duration+interval f0
	endfor
	
	for x from 1 to npoints
		normtime = x / npoints
		select PitchTier normactutimef0
		f0 = Get value at time... normtime+interval
		select TableOfReal 'tableOfReal1$'
		nrows = Get number of rows
		Set value... nrows 1 x+interval*npoints
		Set value... nrows 2 f0
		Set row label (index)... nrows 'label$'
		Insert row (index)... nrows + 1
		select TableOfReal 'tableOfReal2$'
		nrows = Get number of rows
		Set value... nrows 1 normtime*duration+start
		if set_initial_time_for_normf0_to_0
			Set value... nrows 1 normtime*duration+start-firstime
		endif
		Set value... nrows 2 f0
		Set row label (index)... nrows 'label$'
		Insert row (index)... nrows + 1
	endfor
endproc

procedure Sampling
	select PitchTier 'name$'
	Create TableOfReal... samplef0 1 2
	Set column label (index)... 1 SampleTime
	Set column label (index)... 2 F0

	select TextGrid 'name$'
	nintervals = Get number of intervals... 1
	nlabels = 0
	found_interval = 0
	nrows = 0
	sampleStart = Get starting point... 1 1
	sampleEnd = Get end point... 1 nintervals
	Create PitchTier... samplef0 sampleStart sampleEnd
	for m from 1 to nintervals
		select TextGrid 'name$'
		label$ = Get label of interval... 1 m
		start = Get starting point... 1 m
		end = Get end point... 1 m
		duration = end - start
		if not label$ = "" and label$ != "'Silence_marker$'" and duration > 0.01
			nlabels += 1
			if found_interval = 0
				found_interval = 1
			endif

			select PitchTier 'name$'
			nsamples = duration * f0_sample_rate + 1
			for x from 0 to nsamples - 1
				select PitchTier 'name$'
				sample_time = start + x/f0_sample_rate
				f0 = Get value at time... sample_time
				select PitchTier samplef0
				Add point... sample_time f0
				select TableOfReal samplef0
				nrows = Get number of rows
				Set value... nrows 1 sample_time
				Set value... nrows 2 f0
				Set row label (index)... nrows 'label$'
				Insert row (index)... nrows + 1
			endfor
		endif
	endfor
	select TableOfReal samplef0
	if nrows > 1
		Remove row (index)... nrows + 1
	endif
	if smoothing_window_width > 0 and found_interval
		call Smooth samplef0 smoothf0 smoothing_window_width * f0_sample_rate
		Down to TableOfReal... Hertz
		Write to headerless spreadsheet file... 'directory$''name$'.smoothf0
	endif
endproc

procedure Differentiation
	select PitchTier smoothf0
	Rename... semitonef0
	Formula... 12 * ln (self) / ln(2); semitone
	Down to TableOfReal... Hertz
	Set column label (index)... 1 time
	Set column label (index)... 2 f0 (st)

	Write to headerless spreadsheet file... 'directory$''name$'.semitonef0
	Remove
	call Smooth "semitonef0" "smoothf0" 0.05 * f0_sample_rate
	Create PitchTier... velocity sampleStart sampleEnd
	Create TableOfReal... velocity 1 2
	Set column label (index)... 1 SampleTime
	Set column label (index)... 2 Velocity(semitone/s)
	
	select TextGrid 'name$'
	nintervals = Get number of intervals... 1
	n_labeled_intervals = 0
	nrows = 0
	for m from 1 to nintervals
		select TextGrid 'name$'
		label$ = Get label of interval... 1 m
		start = Get starting point... 1 m
		end = Get end point... 1 m
		interval_dur = end - start
		if not label$ = "" and label$ != "'Silence_marker$'" and interval_dur >= 0.001
			select PitchTier smoothf0
			index_first = Get high index from time... start
			index_last = Get low index from time... end
			interval_length = 0
			for x from index_first to index_last - 1
				if x = index_first or x = index_last - 1
					x2 = x + 1
				else
					x2 = x + 2
				endif
				select PitchTier smoothf0
				f01 = Get value at index... x
				f02 = Get value at index... x2
				sampletime1 = Get time from index... x
				sampletime2 = Get time from index... x2
				velocity = (f02 - f01) / (sampletime2 - sampletime1)
				velocity_time = 0.5 * (sampletime1 + sampletime2)
				select PitchTier velocity
				Add point... velocity_time velocity
				interval_length += 1
				select TableOfReal velocity
				nrows = Get number of rows
				Set value... nrows 1 velocity_time
				Set value... nrows 2 velocity
				Set row label (index)... nrows 'label$'
				Insert row (index)... nrows + 1
			endfor
			n_labeled_intervals += 1
			label$[n_labeled_intervals] = label$
			interval_length'n_labeled_intervals' = interval_length
		endif
	endfor
	call Smooth velocity smoothvelocity smoothing_window_width*f0_sample_rate
	Down to TableOfReal... Hertz
	row = 0
	for m from 1 to n_labeled_intervals
		for x from 1 to interval_length'm'
			row += 1
			new_label$ = replace$(label$[m], " ", "_", 0)
			Set row label (index)... row 'new_label$'
		endfor
	endfor
endproc

procedure Means
	select Sound 'name$'
	To Intensity... 100 0 yes
	select TextGrid 'name$'
	nintervals = Get number of intervals... 1
	ntiers = Get number of tiers
	Copy... 'name$'_new_duration

	Create TableOfReal... means 1 18
	Set column label (index)... 1 maxf0
	Set column label (index)... 2 minf0
	Set column label (index)... 3 excursion_size
	Set column label (index)... 4 meanf0
	Set column label (index)... 5 finalf0
	Set column label (index)... 6 mean_intensity
	Set column label (index)... 7 duration
	Set column label (index)... 8 max_velocity
	Set column label (index)... 9 final_velocity
	Set column label (index)... 10 initialf0
	Set column label (index)... 11 target_slope
	Set column label (index)... 12 target_height
	Set column label (index)... 13 target_strength
	Set column label (index)... 14 target_duration
	Set column label (index)... 15 rmse
	Set column label (index)... 16 correlation
	Set column label (index)... 17 mean_rmse
	Set column label (index)... 18 grand_correlation

	interval = 0
	for mm from 1 to nintervals
		select TextGrid 'name$'
		label$ = Get label of interval... 1 mm
		if not label$ = "" and label$ != "'Silence_marker$'" and interval_dur >= 0.001
			interval = interval + 1
			start = Get starting point... 1 mm
			end = Get end point... 1 mm
			select PitchTier semitonef0
			if interval == 1
				index_initial = Get high index from time... start
			endif
			index_final = Get low index from time... end
			duration = 1000 * (end - start)
			start1 = start + perturbation_length/1000
			select TableOfReal means
			Set row label (index)... interval 'label$'
			select PitchTier 'name$'
			meanf0 = Get mean (points)... start1 end
			early_end = end + final_offset
			finalf0 = Get value at time... early_end
			To Pitch... 0.02 30 600
			max_f0 = Get maximum... start1 end Hertz Parabolic
			min_f0 = Get minimum... start1 end Hertz Parabolic
			excursionsize = hertzToSemitones(max_f0) - hertzToSemitones(min_f0)
			Remove
			select Intensity 'name$'
			intensity = Get mean... start end energy
			select PitchTier smooth_velocity
			final_velocity'interval' = Get value at time... early_end
			
			mid = 0.5 * (start + end)
			select PitchTier smooth_velocity
			index_first = Get high index from time... mid
			index_last = Get low index from time... end
			maxvelocity = 0
			for x from index_first to index_last
				v = Get value at index... x
				if abs(v) > abs(maxvelocity)
					maxvelocity = v
				endif
			endfor

			select TableOfReal means
			Set value... interval 1 max_f0
			Set value... interval 2 min_f0
			Set value... interval 3 excursionsize
			Set value... interval 4 meanf0
			Set value... interval 5 finalf0
			Set value... interval 6 intensity
			Set value... interval 7 duration
			Set value... interval 8 maxvelocity
			Set value... interval 9 final_velocity'interval'
			Insert row (index)... interval + 1
		endif
	endfor

	if found_interval
		if task == 4
###################################
;			if !found_interval
				select Sound 'name$'
				tInitial = Get start time
				tFinal = Get end time
;			endif
			call qTA_synthesis

			select Sound 'name$'
			To Manipulation... 0.01 75 600
			Extract duration tier
			Rename... 'name$'
		
			tInitial_new_dur = tInitial
			tFinal_new_dur = tFinal
			interval = 0
			select TextGrid 'name$'
			for n from 1 to nintervals
				select TextGrid 'name$'
				label$ = Get label of interval... 1 n
				if not label$ = "" and label$ != "'Silence_marker$'" and interval_dur >= 0.001
					interval += 1
					if dur_ratio'interval' > 1.001 or dur_ratio'interval' < 0.999
						select TextGrid 'name$'
						start = Get starting point... 1 n
						end = Get end point... 1 n
						select DurationTier 'name$'
						if not hasnewduration
							Add point... start 1.0
							last_boundary = start
						endif
						Add point... start+0.0001 dur_ratio'interval'
						Add point... end-0.0001 dur_ratio'interval'
						if n = nintervals-1
							Add point... end 1
						endif
						hasnewduration = 1
					else
						select DurationTier 'name$'
						Add point... end 1
					endif
					if hasnewduration
						select TextGrid 'name$'_new_duration
						new_boundary = last_boundary+dur_ratio'interval'*(end-start)
						last_boundary = new_boundary
						if new_boundary + end-start > tFinal
							Extend time: (new_boundary + end - start) - tFinal, "end"
							tFinal = new_boundary + end - start
						endif
						for tierN from 1 to ntiers-1
							current_label$ = Get label of interval... tierN n
							next_label$ = Get label of interval... tierN n+1
							Remove right boundary: tierN, n
							Insert boundary: tierN, new_boundary
							Set interval text: tierN, n, current_label$
							Set interval text: tierN, n+1, next_label$
						endfor
						tFinal_new_dur = new_boundary
					endif
				endif
			endfor
			hasnewduration = 0
			select DurationTier 'name$'
			plus Manipulation 'name$'
			Replace duration tier
			select PitchTier fittedf0_st
			Copy... fittedf0_hz
			Formula... exp(self*ln(2)/12)
			plus Manipulation 'name$'
			Replace pitch tier
			select Manipulation 'name$'
			Get resynthesis (overlap-add)
			Rename... 'name$'_qTA
;			select Manipulation 'name$'
;			Remove
###################################
		else
			call Target
			select Sound 'name$'
			To Manipulation... 0.01 75 600
			select PitchTier fittedf0_st
			Copy... fittedf0_hz
			Formula... exp(self*ln(2)/12)
			plus Manipulation 'name$'
			Replace pitch tier
		endif
		select PitchTier fittedf0_st
		call correlation index_initial index_final
		if save_synthetic_sounds
			select Manipulation 'name$'
			Get resynthesis (overlap-add)
;			select Sound 'name$'_qTA
			createDirectory: "'directory$'/'qTA_synthesis_folder$'"
			Save as WAV file: "'directory$'/'qTA_synthesis_folder$'/'name$'_qTA.wav"
;			Remove
			select TextGrid 'name$'_new_duration
			Write to short text file: "'directory$'/'qTA_synthesis_folder$'/'name$'_qTA.target"
		endif
		select TableOfReal fittedf0_st
		Write to headerless spreadsheet file... 'directory$''name$'.qTAf0
		Remove
		select TableOfReal fittedvelocity
		Write to headerless spreadsheet file... 'directory$''name$'.qTAvelocity
		plus PitchTier fittedvelocity
		if task != 4
			plus TableOfReal trainingf0
		endif
		Remove

		interval = 0
		if task != 4
			for mm from 1 to nintervals
				select TextGrid 'name$'
				label$ = Get label of interval... 1 mm
				start = Get starting point... 1 mm
				end = Get end point... 1 mm
				interval_dur = end - start
				if not label$ = "" and label$ != "'Silence_marker$'" and interval_dur >= 0.001
					interval = interval + 1
					select TableOfReal means
					Set value... interval 10 xInitial
					Set value... interval 11 mBest'interval'
					Set value... interval 12 bBest'interval'
					Set value... interval 13 lambdaBest'interval'
					Set value... interval 14 duration'interval'
					Set value... interval 15 rmse'interval'
					Set value... interval 16 corr'interval'
					Set value... interval 17 mean_rmse/nintervals
					Set value... interval 18 corr

					select TextGrid 'name$'
					for nn from 3 to ntiers-1
						has_start_point = 0
						has_end_point = 0
						nintervals_current_tier = Get number of intervals... nn
						for ii from 1 to nintervals_current_tier
							start_point = Get starting point... nn ii
							end_point = Get end point... nn ii
							if abs(start_point - start) < 0.001
								has_start_point = 1
							endif
							if abs(end_point - end) < 0.001
								has_end_point = 1
							endif
						endfor
						if !has_start_point
							Insert boundary: nn, start
						endif
						if !has_end_point
							Insert boundary: nn, end
						endif
					endfor
					
					select TextGrid 'name$'
					if interval = 1
						Set interval text... 4 1 'xInitial'
					endif

					slope = mBest'interval'
					height = bBest'interval'
					strength = lambdaBest'interval'
					duration = 1000 * duration'interval'
					Set interval text... 3 mm 'slope'
					Set interval text... 4 mm 'height'
					Set interval text... 5 mm 'strength'
					Set interval text... 6 mm 'duration'
				endif
			endfor
		endif
	endif
	select TableOfReal means
	nrows = Get number of rows
	if nrows > 1
		Remove row (index)... nrows
	endif
endproc

procedure All_means file_name$
	Read TableOfReal from headerless spreadsheet file... 'directory$''name$'.means
	nrows = Get number of rows
	ncolumns = Get number of columns
	titleline$ = "Filename"
	target_titleline$ = "Filename	initialf0	mean_rmse	grand_correlation"
	maxf0line$ = name$
	minf0line$ = name$
	excursionsizeline$ = name$
	meanf0line$ = name$
	finalf0line$ = name$
	intensityline$ = name$
	durationline$ = name$
	maxvelocityline$ = name$
	finalvelocityline$ = name$
	targetline$ = name$
	for n from 1 to nrows
		if !hasmeanstitle
			rowname$ = Get row label... n
			titleline$ = "'titleline$'	'rowname$'"
			target_titleline$ = "'target_titleline$'	'rowname$'__slope	height	strength	duration	rmse	correlation"
		endif
		maxf0 = Get value... n 1
		maxf0line$ = "'maxf0line$'	'maxf0'"
		minf0 = Get value... n 2
		minf0line$ = "'minf0line$'	'minf0'"
		excursionsize = Get value... n 3
		excursionsizeline$ = "'excursionsizeline$'	'excursionsize'"
		meanf0 = Get value... n 4
		meanf0line$ = "'meanf0line$'	'meanf0'"
		finalf0 = Get value... n 5
		finalf0line$ = "'finalf0line$'	'finalf0'"
		intensity = Get value... n 6
		intensityline$ = "'intensityline$'	'intensity'"
		duration = Get value... n 7
		durationline$ = "'durationline$'	'duration'"
		maxvelocity = Get value... n 8
		maxvelocityline$ = "'maxvelocityline$'	'maxvelocity'"
		final_velocity = Get value... n 9
		finalvelocityline$ = "'finalvelocityline$'	'final_velocity'"
		xInitial = Get value... n 10
		target_slope = Get value... n 11
		target_height = Get value... n 12
		strength = Get value... n 13
		target_duration = Get value... n 14
		rmse = Get value... n 15
		corr = Get value... n 16
		if ncolumns > 16
			mean_rmse = Get value... n 17
			grand_corr = Get value... n 18
		else
			mean_rmse = -1
			grand_corr = -1
		endif
		if n = 1
			targetline$ = "'targetline$'	'xInitial'	'mean_rmse'	'grand_corr'	'target_slope'	'target_height'	'strength'	'target_duration'	'rmse'	'corr'"
		else
			targetline$ = "'targetline$'	'target_slope'	'target_height'	'strength'	'target_duration'	'rmse'	'corr'"
		endif
	endfor
	if !hasmeanstitle
		filedelete maxf0.txt
		filedelete minf0.txt
		filedelete excursionsize.txt
		filedelete meanf0.txt
		filedelete finalf0.txt
		filedelete meanintensity.txt
		filedelete duration.txt
		filedelete maxvelocity.txt
		filedelete finalvelocity.txt
		filedelete targets.txt
		titleline$ = "'titleline$''newline$'"
		target_titleline$ = "'target_titleline$''newline$'"
		fileappend maxf0.txt 'titleline$'
		fileappend minf0.txt 'titleline$'
		fileappend excursionsize.txt 'titleline$'
		fileappend meanf0.txt 'titleline$'
		fileappend finalf0.txt 'titleline$'
		fileappend meanintensity.txt 'titleline$'
		fileappend duration.txt 'titleline$'
		fileappend maxvelocity.txt 'titleline$'
		fileappend finalvelocity.txt 'titleline$'
		fileappend targets.txt 'target_titleline$'
		hasmeanstitle = 1
	endif
	maxf0line$ = "'maxf0line$''newline$'"
	fileappend "maxf0.txt" 'maxf0line$'
	minf0line$ = "'minf0line$''newline$'"
	fileappend "minf0.txt" 'minf0line$'
	excursionsizeline$ = "'excursionsizeline$''newline$'"
	fileappend "excursionsize.txt" 'excursionsizeline$'
	meanf0line$ = "'meanf0line$''newline$'"
	fileappend "meanf0.txt" 'meanf0line$'
	finalf0line$ = "'finalf0line$''newline$'"
	fileappend "finalf0.txt" 'finalf0line$'
	intensityline$ = "'intensityline$''newline$'"
	fileappend "meanintensity.txt" 'intensityline$'
	durationline$ = "'durationline$''newline$'"
	fileappend "duration.txt" 'durationline$'
	maxvelocityline$ = "'maxvelocityline$''newline$'"
	fileappend "maxvelocity.txt" 'maxvelocityline$'
	finalvelocityline$ = "'finalvelocityline$''newline$'"
	fileappend "finalvelocity.txt" 'finalvelocityline$'
	targetline$ = "'targetline$''newline$'"
	fileappend "targets.txt" 'targetline$'
	Remove
endproc

procedure All_normf0 file_name$
	Read TableOfReal from headerless spreadsheet file... 'directory$''name$'.normtimef0
	nrows = Get number of rows
	titleline$ = "Normtime"
	resultline$ = "_"+name$
	for n from 1 to nrows
		if !hasnormf0
			normtime$ = Get value... n 1
			titleline$ = "'titleline$'	'normtime$'"
		endif
		value = Get value... n 2
		resultline$ = "'resultline$'	'value'"
	endfor
	if !hasnormf0
		filedelete normtimef0.txt
		titleline$ = "'titleline$''newline$'"
		fileappend normtimef0.txt 'titleline$'
		hasnormf0 = 1
	endif
	resultline$ = "'resultline$''newline$'"
	fileappend "normtimef0.txt" 'resultline$'
	Remove
	Read TableOfReal from headerless spreadsheet file... 'directory$''name$'.actutimenormf0
	nrows = Get number of rows
	titleline$ = "Norm actual time"
	resultline$ = "_"+name$
	for n from 1 to nrows
		if !hasnormactutime
			titleline$ = "'titleline$'	'n'"
		endif
		value = Get value... n 1
		resultline$ = "'resultline$'	'value'"
	endfor
	if !hasnormactutime
		filedelete normactutime.txt
		titleline$ = "'titleline$''newline$'"
		fileappend normactutime.txt 'titleline$'
		hasnormactutime = 1
	endif
	resultline$ = "'resultline$''newline$'"
	fileappend "normactutime.txt" 'resultline$'
	Remove
endproc

procedure All_samplef0 file_name$
	Read TableOfReal from headerless spreadsheet file... 'directory$''name$'.samplef0
	nrows = Get number of rows
	titleline$ = "Sampletime"
	resultline$ = "_"+name$
	for n from 1 to nrows
		if !hassamplef0
			sampletime = Get value... n 1
			if n == 1
				onsettime = sampletime
			endif
			sampletime = sampletime - onsettime
			titleline$ = "'titleline$'	'sampletime'"
		endif
		value = Get value... n 2
		resultline$ = "'resultline$'	'value'"
	endfor
	if !hassamplef0
		filedelete samplef0.txt
		titleline$ = "'titleline$''newline$'"
		fileappend samplef0.txt 'titleline$'
		hassamplef0 = 1
	endif
	resultline$ = "'resultline$''newline$'"
	fileappend "samplef0.txt" 'resultline$'
	Remove
endproc

procedure All_f0velocity file_name$
	Read TableOfReal from headerless spreadsheet file... 'directory$''name$'.f0velocity
	nrows = Get number of rows
	titleline$ = "Sampletime"
	resultline$ = "_"+name$
	for n from 1 to nrows
		if !hasf0velocity
			sampletime = Get value... n 1
			if n == 1
				onsettime = sampletime
			endif
			sampletime = sampletime - onsettime
			titleline$ = "'titleline$'	'sampletime'"
		endif
		value = Get value... n 2
		resultline$ = "'resultline$'	'value'"
	endfor
	if !hasf0velocity
		filedelete f0velocity.txt
		titleline$ = "'titleline$''newline$'"
		fileappend f0velocity.txt 'titleline$'
		hasf0velocity = 1
	endif
	resultline$ = "'resultline$''newline$'"
	fileappend "f0velocity.txt" 'resultline$'
	Remove
endproc

procedure All_qTAf0 file_name$
	Read TableOfReal from headerless spreadsheet file... 'directory$''name$'.qTAf0
	nrows = Get number of rows
	titleline$ = "Sampletime"
	resultline$ = "_"+name$
	for n from 1 to nrows
		if !hasqTAf0
			sampletime = Get value... n 1
			if n == 1
				onsettime = sampletime
			endif
			sampletime = sampletime - onsettime
			titleline$ = "'titleline$'	'sampletime'"
		endif
		value = Get value... n 2
		resultline$ = "'resultline$'	'value'"
	endfor
	if !hasqTAf0
		filedelete f0velocity.txt
		titleline$ = "'titleline$''newline$'"
		fileappend qTAf0.txt 'titleline$'
		hasqTAf0 = 1
	endif
	resultline$ = "'resultline$''newline$'"
	fileappend "qTAf0.txt" 'resultline$'
	Remove
endproc

procedure Smooth curve_in$ curve_out$ width
	Create PitchTier... 'curve_out$' sampleStart sampleEnd
	
	for j from 1 to width							; make a triangular window of size = width 
		if j < width / 2 + 0.5
			weight'j' = j
		else 
			weight'j' = width - j + 1
		endif
	endfor

	select PitchTier 'curve_in$'
	smooth_end = Get number of points

	for i from 1 to width / 2						; smooth initial points: 0 to width/2 - 1 
	    n = 0.0
		smoothsample = 0.0
		select PitchTier 'curve_in$'
		sample_time = Get time from index... i
	    for j from 1 to width/2 + i					; window size = width/2 to width - 1
	    	j = 'j:0'
	    	rawsample = Get value at index... j
	    	index = width / 2 + j - i
	    	index = 'index:0'
			smoothsample += weight'index' * rawsample
			n += weight'index'
	    endfor
		smoothsample /= n
		if smoothsample != undefined
			select PitchTier 'curve_out$'
			Add point... sample_time smoothsample
		endif
	endfor
	
	for i from width/2 to smooth_end - width/2				; smooth from width/2 to end-width/2
		n = 0
		smoothsample = 0.0
		select PitchTier 'curve_in$'
		sample_time = Get time from index... i
		for j from 1 to width
	    	rawsample = Get value at index... i-width/2+j
			smoothsample += weight'j' * rawsample
			n += weight'j'
		endfor
		smoothsample /= n
		if smoothsample != undefined
			select PitchTier 'curve_out$'
			Add point... sample_time smoothsample
		endif
	endfor
	
	i = width/2
	while i > 0										; smooth final points: end - width/2 to end
		n = 0.0
		smoothsample = 0.0
		select PitchTier 'curve_in$'
		sample_time = Get time from index... smooth_end-i
		j = width/2 + i
		j = 'j:0'
		while j > 1									; window size = width - 1 to width/2 
	    	rawsample = Get value at index... smooth_end-j
	    	index = width/2+i-j + 1
	    	index = 'index:0'
			smoothsample += weight'index' * rawsample
			n += weight'index'
			j -= 1
		endwhile
		smoothsample /= n
		if smoothsample != undefined
			select PitchTier 'curve_out$'
			Add point... sample_time smoothsample
		endif
		i -= 1
	endwhile
endproc

######################################################################################
procedure correlation index_initial index_final
        len =  1 + index_final - index_initial
        ee = 1e-20
        mean_rawf0 = 0
        mean_qTAf0 = 0
        ssxx = 0
        ssyy = 0
        ssxy = 0
        
        for x from index_initial to index_final
			select PitchTier semitonef0
			sampletime = Get time from index... x
			rawf0 = Get value at time... sampletime
			select PitchTier fittedf0_st
			qTAf0 = Get value at time... sampletime
			mean_rawf0 += rawf0 / len
			mean_qTAf0 += qTAf0 / len
        endfor
        
        for x from index_initial to index_final
			select PitchTier semitonef0
			sampletime = Get time from index... x
			rawf0 = Get value at time... sampletime
			select PitchTier fittedf0_st
			qTAf0 = Get value at time... sampletime
			ssxx += (rawf0 - mean_rawf0 + ee) * (rawf0 - mean_rawf0 + ee)
			ssyy += (qTAf0 - mean_qTAf0 + ee) * (qTAf0 - mean_qTAf0 + ee)
			ssxy += (rawf0 - mean_rawf0 + ee) * (qTAf0 - mean_qTAf0 + ee)
        endfor
        corr = ssxy * ssxy / (ssxx * ssyy)
        corr = sqrt(corr)
endproc
######################################################################################

procedure Target
	Create PitchTier... fittedf0_st sampleStart sampleEnd
	Create PitchTier... fittedvelocity sampleStart sampleEnd
	Create TableOfReal... fittedf0_st 1 2
	Set column label (index)... 1 SampleTime
	Set column label (index)... 2 fittedf0 (st)
	Create TableOfReal... fittedvelocity 1 2
	Set column label (index)... 1 SampleTime
	Set column label (index)... 2 velocity (st/s)

	filedelete 'directory$''name$'.primitive_code
	
	select TextGrid 'name$'
	nintervals = Get number of intervals... 1
	nTiers = Get number of tiers
	if nTiers > 1
		tier2$ = Get tier name... 2
		nintervals2 = Get number of intervals... 2
	else
		tier2$ = "null"
	endif
	if nTiers < 2 or tier2$ != "target" or tier2$ == "target" and nintervals2 != nintervals
		Duplicate tier... 1 2 target
		Replace interval text... 2 0 nintervals . "" Regular Expressions
		if tier2$ == "target" and nintervals2 != nintervals
			Remove tier... 3
		endif
		Write to short text file... 'directory$''name$'.target
	endif
	nrows = 0
	interval_t = 0
	mean_rmse = 0
;	silence_duration = 0

	select PitchTier semitonef0
	Down to TableOfReal... Hertz
	Rename... trainingf0

	for nm from 1 to nintervals
		select TextGrid 'name$'
		label$ = Get label of interval... 1 nm
		target$ = Get label of interval... 2 nm
		start = Get starting point... 1 nm
		end = Get end point... 1 nm
		interval_dur = end - start

		if not label$ = "" and label$ != "'Silence_marker$'" and interval_dur >= 0.001
# Interval has to be longer than 0.01 s
			interval_t = interval_t + 1
			end_last = end
			if start - end_last > 0.01
				d_sil = start - end_last
				v_start = start
				s_start = end_last
				select PitchTier semitonef0
				row_index = Get low index from time... s_start
				s_start = Get time from index... row_index
				row_index2 = Get high index from time... v_start
				v_start = Get time from index... row_index2
				f01 = Get value at time... s_start
				f02 = Get value at time... v_start
			else
				d_sil = -1
				f01 = 0
				f02 = 0
			endif


			select PitchTier semitonef0
			index_first = Get low index from time... start
			index_last = Get high index from time... end
			tFinal = end
			if interval_t == 1
				tInitial = Get time from index... index_first
				xInitial = Get value at index... index_first
				x0 = xInitial
				v0 = 0
				a0 = 0
				silence_duration = 0
				fileappend 'directory$''name$'.primitive_code 'xInitial''newline$'
			elsif silence_duration > minimum_pause_duration / 1000
				x0 = Get value at index... index_first
				v0 = 0
				a0 = 0
				silence_duration = 0
			else
				x0 = xFinal
				v0 = vFinal
				a0 = aFinal
			endif
			
			mMin = min_target_slope
			mMax = max_target_slope
			if !ignore_target_labels_in_Tier_2
				if index_regex(target$,"\d")	; if the label is a numerical value
					slope = 'target$'
					if slope < 0
						mMin = slope
						mMax = 0
					elsif slope > 0
						mMax = slope
						mMin = 0
					else
						mMin = 0
						mMax = 0.01
					endif
				elsif left$(target$,1) == "h" or left$(target$,1) == "H" or left$(target$,1) == "l" or left$(target$,1) == "L" or left$(target$,1) == "m" or left$(target$,1) == "M"
					mMin = 0
					mMax = 0.01
				elsif left$(target$,1) == "f" or left$(target$,1) == "F" or final_velocity'interval_t' < 0 and use_final_velocity_to_constrain_slope
					mMax = 0
				elsif left$(target$,1) == "r" or left$(target$,1) == "R" or final_velocity'interval_t' > 0 and use_final_velocity_to_constrain_slope
					mMin = 0
				endif
			endif
			bMin = min_target_height
			bMax = max_target_height
			lambdaMin = min_strength
			lambdaMax = max_strength

			filedelete 'directory$'config
			fileappend 'directory$'config 'mMin' 'mMax''newline$'
			fileappend 'directory$'config 'bMin' 'bMax''newline$'
			fileappend 'directory$'config 'lambdaMin' 'lambdaMax' 'fixed_strength''newline$'
			fileappend 'directory$'config 'x0' 'v0' 'a0' 'newline$'
####
			fileappend 'directory$'config 'd_sil' 'f01' 'f02'
####
			
			select TableOfReal trainingf0
			index_last1 = index_last-1
			Extract row ranges... 'index_first':'index_last1'
			Write to short text file... 'directory$'data
			Remove

			if b_Location = 1
				if fileReadable("./learnqta")
					system ./learnqta_onset 'directory$'
				elsif fileReadable(".\learnqta")
					system .\learnqta_onset 'directory$'
				else
					printline Cannot find learnqta!
					exit
				endif	
			else
				if fileReadable("./learnqta")
					system ./learnqta ./
				elsif fileReadable(".\learnqta.exe")
					system .\learnqta .\
				else
					printline Cannot find learnqta!
					exit
				endif	
			endif
			buff$ < output
			
	
	
			#First line: qTA parameters
			len_buff = length(buff$)
			line_index = index(buff$,newline$)
			sbuff$ = left$(buff$,line_index-1)
			buff$ = mid$(buff$,line_index+1,len_buff)
			len_sbuff = length(sbuff$)
	
			#First line: m
			sep_index = index(sbuff$," ")
			pbuff$ = left$(sbuff$,sep_index-1)
			sbuff$ = mid$(sbuff$,sep_index+1,len_sbuff)
			mBest'interval_t' = 'pbuff$'
			m = mBest'interval_t'
	
			#First line: b
			sep_index = index(sbuff$," ")
			pbuff$ = left$(sbuff$,sep_index-1)
			sbuff$ = mid$(sbuff$,sep_index+1,len_sbuff)
			if b_Location = 1
				bBest'interval_t' = 'pbuff$' - xInitial
			else
				bBest'interval_t' = 'pbuff$' + m * (end-start) - xInitial
			endif
			b = bBest'interval_t' + xInitial
	
			#First line: lambda
			lambdaBest'interval_t' = 'sbuff$'
			lambda = lambdaBest'interval_t'
	
			duration'interval_t' = end - start
			d = duration'interval_t'
	
			fileappend 'directory$''name$'.primitive_code 'm', 'b', 'lambda', 'd'
			if interval_t < nlabels
				fileappend 'directory$''name$'.primitive_code :
			endif
			
			#Second line: final F0 articulatory state
			len_buff = length(buff$)
			line_index = index(buff$,newline$)
			sbuff$ = left$(buff$,line_index-1)
			buff$ = mid$(buff$,line_index+1,len_buff)
			len_sbuff = length(sbuff$)
	
			#Second line: xFinal
			sep_index = index(sbuff$," ")
			pbuff$ = left$(sbuff$,sep_index-1)
			sbuff$ = mid$(sbuff$,sep_index+1,len_sbuff)
			xFinal = 'pbuff$'
	
			#Second line: vFinal
			sep_index = index(sbuff$," ")
			pbuff$ = left$(sbuff$,sep_index-1)
			sbuff$ = mid$(sbuff$,sep_index+1,len_sbuff)
			vFinal = 'pbuff$'
	
			#Second line: aFinal
			aFinal = 'sbuff$'
	
			#Third line: rmse and correlation results
			len_buff = length(buff$)
			line_index = index(buff$,newline$)
			sbuff$ = left$(buff$,line_index-1)
			buff$ = mid$(buff$,line_index+1,len_buff)
			len_sbuff = length(sbuff$)
	
			#Third line: rmse
			sep_index = index(sbuff$," ")
			pbuff$ = left$(sbuff$,sep_index-1)
			sbuff$ = mid$(sbuff$,sep_index+1,len_sbuff)
			rmse'interval_t' = 'pbuff$'
############
			mean_rmse += rmse'interval_t'
	
			#Third line: correlation
			corr'interval_t' = 'sbuff$'
	
			#Forth line: number of sample
			len_buff = length(buff$)
			line_index = index(buff$,newline$)
			sbuff$ = left$(buff$,line_index-1)
			buff$ = mid$(buff$,line_index+1,len_buff)
			len_sbuff = length(sbuff$)
			number_of_row = 'sbuff$'
	
			#Remaining: time and synthesized f
			select PitchTier fittedf0_st
			for i from 1 to number_of_row
				select TableOfReal fittedf0_st
				len_buff = length(buff$)
				line_index = index(buff$,newline$)
				sbuff$ = left$(buff$,line_index-1)
				buff$ = mid$(buff$,line_index+1,len_buff)
				len_sbuff = length(sbuff$)
				#printline 'sbuff$'
	
				sep_index = index(sbuff$," ")
				pbuff$ = left$(sbuff$,sep_index-1)
				sbuff$ = mid$(sbuff$,sep_index+1,len_sbuff)
				sampletime = 'pbuff$'
				if i > 1
					qTAf0_previous = qTAf0
				endif
				qTAf0 = 'sbuff$'
	
				nrows = Get number of rows
				Set value... nrows 1 sampletime
				Set value... nrows 2 qTAf0
				Set row label (index)... nrows 'label$'
				Insert row (index)... nrows + 1
				select PitchTier fittedf0_st
				Add point... sampletime qTAf0
				if i > 1
					select TableOfReal fittedvelocity
					nrows = Get number of rows
					Set value... nrows 1 sampletime
					Set value... nrows 2  (qTAf0 - qTAf0_previous) * f0_sample_rate
					Set row label (index)... nrows 'label$'
					Insert row (index)... nrows + 1
				endif
			endfor

			for x from index_first to index_last - 1
				select PitchTier semitonef0
				sampletime = Get time from index... x
				rawf0 = Get value at time... sampletime
			endfor
		else
			start = Get starting point... 1 nm
			end = Get end point... 1 nm
			silence_duration = end - start			
		endif
	endfor
	filedelete 'directory$'config
	filedelete 'directory$'data
	filedelete 'directory$'output
	select TableOfReal fittedf0_st
	if nrows > 1
		Remove row (index)... nrows + 1
	endif
	select TableOfReal fittedvelocity
	nrows = Get number of rows
	if nrows > 1
		Remove row (index)... nrows
	endif
endproc

procedure plot_target t1 t2
	demoWindowTitle ("qTAtrainer 'version$'")
	demo Erase all
	demo Blue
	demo Times
	demo Line width... 0.001
	demo 12
	demo Select outer viewport... 0 100 50 100
	if found_interval
		select PitchTier semitonef0
		To Pitch... 0.02 60 600
	else
		select Sound 'name$'
		To Pitch (ac)... 0 75 15 no 0.03 0.45 0.01 0.35 0.14 600
		Rename... semitonef0
		Formula... 12 * ln (self) / ln(2); semitone
		Down to PitchTier
	endif
	select Pitch semitonef0
	f0_floor = Get minimum... 0 0 Hertz Parabolic
	f0_ceiling = Get maximum... 0 0 Hertz Parabolic
	f0_floor -= 10
	f0_ceiling += 10
	select PitchTier semitonef0
	demo Draw... t1 t2 f0_floor f0_ceiling no
	demo Line width... 0.5
	demo Draw inner box
	if found_interval
		select PitchTier fittedf0_st
		To Pitch... 0.02 60 600
		demo Black
		demo Text special... t1-0.04*(t2-t1) Centre 0.5*(f0_floor+f0_ceiling) Bottom Times 12 90 Pitch (st)
		demo Text special... 0.5*(t1+t2) Centre f0_floor-0.125*(f0_ceiling-f0_floor) Top Times 12 0 Time (s)
		demo Red
		demo Line width... 3
		demo Solid line
		if task == 4
			demo Draw... tInitial_new_dur tFinal_new_dur f0_floor f0_ceiling no
		else
			demo Draw... t1 t2 f0_floor f0_ceiling no
		endif
		Remove
	endif
	demo Line width... 0.5
	demo Marks bottom every... 1 0.1 no yes no
	demo Marks bottom every... 1 0.5 yes yes no
	demo Marks left every... 1 5 no yes no
	demo Marks left every... 1 10 yes yes no

	interval_plot = 0
	last_end = 32767
	total_dur_change = 0
	demo Dashed line
	select TextGrid 'name$'
	nintervals = Get number of intervals... 1
	if task == 4
		xInitial = xInitialNew
	endif
	for nn from 1 to nintervals
		select TextGrid 'name$'
		label$ = Get label of interval... 1 nn
		start = Get starting point... 1 nn
		end = Get end point... 1 nn
		interval_dur = end - start
		if not label$ = "" and label$ != "'Silence_marker$'" and interval_dur >= 0.001
			interval_plot += 1
			if task = 4
				m = mNew'interval_plot'
				b = bNew'interval_plot'
				lambda = lambdaNew'interval_plot'
				duration = durationNew'interval_plot'
				start = Get starting point... 1 nn
				start += total_dur_change
				end = start + duration
				total_dur_change += dur_change'interval_plot'
			else
				start = Get starting point... 1 nn
				end = Get end point... 1 nn
				duration = end - start
				m = mBest'interval_plot'
				b = bBest'interval_plot'
				lambda = lambdaBest'interval_plot'
			endif
			strength = 3 * lambda / max_strength
			if strength <= 0
				strength = 0.001
			endif
			y1 = xInitial + b - m*duration
			b -= m*duration
			y2 = xInitial + b + m*duration
			demo Black
			demo Text special... start Left f0_ceiling Bottom Times 12 0 'label$'
			demo Lime
			demo Line width... strength
			demo Draw line... start y1 end y2
			demo Grey
			demo Line width... 0.15
			demo Draw line... start f0_floor start f0_ceiling
			if (start - last_end) > 0.001
				demo Paint rectangle... white last_end start f0_floor+1 f0_ceiling-1
				demo Draw line... last_end f0_floor last_end f0_ceiling
				demo Silver
			endif
			last_end = end
		endif
	endfor
	if task = 4
		demo Blue
		select PitchTier semitonef0
		demo Draw... t1 t2 f0_floor f0_ceiling no
	endif
	
	demo Select inner viewport... 7 95 42 45
	demo Axes... 7 95 42 45
	demo Paint rounded rectangle... {0.98,0.98,0.8} 5 10 42 45 2	; plot
	demo Paint rounded rectangle... {0.98,0.98,0.8} 12 18 42 45 2	; zoom in
	demo Paint rounded rectangle... {0.98,0.98,0.8} 19 25 42 45 2	; zoom out
	demo Paint rounded rectangle... {0.98,0.98,0.8} 26 32 42 45 2	; zoom all
	demo Paint rounded rectangle... {0.98,0.98,0.8} 34 38 42 45 2	; shift to start
	demo Paint rounded rectangle... {0.98,0.98,0.8} 39 43 42 45 2	; shift left
	demo Paint rounded rectangle... {0.98,0.98,0.8} 44 48 42 45 2	; shif right
	demo Paint rounded rectangle... {0.98,0.98,0.8} 49 53 42 45 2	; shift to end
	demo Paint rounded rectangle... {0.98,0.98,0.8} 55 63 42 45 2	; play original
	demo Paint rounded rectangle... {0.98,0.98,0.8} 64 74 42 45 2	; play resynthesis
	demo Paint rounded rectangle... {0.98,0.98,0.8} 76 82 42 45 2	; Previous
	demo Paint rounded rectangle... {0.98,0.98,0.8} 83 87 42 45 2	; Next
	demo Paint rounded rectangle... {0.98,0.98,0.8} 89 93 42 45 2	; Exit
	demo Solid line
	demo Line width... 0.5
	demo Black
	demo Draw rounded rectangle... 5 10 42 45 2
	demo Draw rounded rectangle... 12 18 42 45 2
	demo Draw rounded rectangle... 19 25 42 45 2
	demo Draw rounded rectangle... 26 32 42 45 2
	demo Draw rounded rectangle... 34 38 42 45 2
	demo Draw rounded rectangle... 39 43 42 45 2
	demo Draw rounded rectangle... 44 48 42 45 2
	demo Draw rounded rectangle... 49 53 42 45 2
	demo Draw rounded rectangle... 55 63 42 45 2
	demo Draw rounded rectangle... 64 74 42 45 2
	demo Draw rounded rectangle... 76 82 42 45 2
	demo Draw rounded rectangle... 83 87 42 45 2
	demo Draw rounded rectangle... 89 93 42 45 2
	demo Text special... 7.5 centre 43.5 half Times 10 0 Replot
	demo Text special... 15 centre 43.5 half Times 10 0 Zoom in
	demo Text special... 22 centre 43.5 half Times 10 0 Zoom out
	demo Text special... 29 centre 43.5 half Times 10 0 Zoom all
	demo Text special... 36 centre 43.5 half Helvetica 10 0 |<<
	demo Text special... 41 centre 43.5 half Helvetica 10 0 <<
	demo Text special... 46 centre 43.5 half Helvetica 10 0 >>
	demo Text special... 51 centre 43.5 half Helvetica 10 0 >>|
	demo Blue
	demo Text special... 59 centre 43.5 half Times 10 0 Play original
	demo Red
	demo Text special... 69 centre 43.5 half Times 10 0 Play resynthesis
	demo Black
	demo Text special... 79 centre 43.5 half Times 10 0 Previous
	demo Text special... 85 centre 43.5 half Times 10 0 Next
	demo Text special... 91 centre 43.5 half Times 10 0 Exit
	demo Text special... 2 left 32 bottom Courier 12 0 Current sound: 'name$' (No.'current_file')
	if more_file = 0
		demo Paint rectangle... White 0 100 25 30
		demo Red
		demo Text special... 2 left 27 bottom Courier 14 0 Last sound file...
	endif
	call on_input
endproc

procedure on_input
	demo Select inner viewport... 0 100 0 100
	demo Axes... 0 100 0 100
	loop = 1
	while (loop and demoWaitForInput ( ))
		if demoClicked ( )
			if demoClickedIn (12, 18, 42, 45)			; clicked Zoom in
				new_window_centre = t1 + (t2-t1)/2
				plot_width = 0.5 * (t2-t1)
				t1 = max (new_window_centre - plot_width/2, tInitial)
				t2 = min (new_window_centre + plot_width/2, tFinal)
				plot_width = t2-t1
				call plot_target t1 t2
			elsif demoClickedIn (19, 25, 42, 45)		; clicked Zoom out
				new_window_centre = t1 + (t2-t1)/2
				plot_width = 2 * (t2-t1)
				t1 = max (new_window_centre - plot_width/2, tInitial)
				t2 = min (new_window_centre + plot_width/2, tFinal)
				plot_width = t2-t1
				call plot_target t1 t2
			elsif demoClickedIn (26, 32, 42, 45)		; clicked Zoom all
				new_window_centre = tInitial + (tFinal-tInitial)/2
				plot_width = 100 * (tFinal-tInitial)
				t1 = max (new_window_centre - plot_width/2, tInitial)
				t2 = min (new_window_centre + plot_width/2, tFinal)
				plot_width = t2-t1
				call plot_target t1 t2
			elsif demoClickedIn (34, 38, 42, 45)			; clicked Shift to start
				call plot_target tInitial t1+plot_width
			elsif demoClickedIn (39, 43, 42, 45)		; clicked Shift left
					t1 = min (t1 - plot_width, tFinal + plot_width)
					t1 = max (t1, tInitial)
					t2 = t1 + plot_width
					call plot_target t1 t2
			elsif demoClickedIn (44, 48, 42, 45)		; clicked Shift right
					t1 = min (t1 + plot_width, tFinal - plot_width)
					t1 = max (t1, tInitial)
					t2 = t1 + plot_width
					call plot_target t1 t2
			elsif demoClickedIn (49, 53, 42, 45)		; clicked Shift to end
					call plot_target tFinal-plot_width tFinal
			elsif demoClickedIn (55, 63, 42, 45)		; clicked Play original
				demo Paint rounded rectangle... Yellow 55 63 42 45 2
				demo Black
				demo Draw rounded rectangle... 55 63 42 45 2
				demo Text special... 59 centre 43.5 half Times 10 0 Play original
				sound = Read from file... 'directory$''file_name$'
				Extract part... t1 t2 Rectangular 1.0 yes
				Play
				plus sound
				Remove
				demo Paint rounded rectangle... {0.98,0.98,0.8} 55 63 42 45 2
				demo Black
				demo Draw rounded rectangle... 55 63 42 45 2
				demo Text special... 59 centre 43.5 half Times 10 0 Play original
			elsif demoClickedIn (64, 74, 42, 45)		; clicked Play resynthesis
				if found_interval
					demo Paint rounded rectangle... Yellow 64 74 42 45 2
					demo Black
					demo Draw rounded rectangle... 64 74 42 45 2
					demo Text special... 69 centre 43.5 half Times 10 0 Play resynthesis
					if task = 4
						select Sound 'name$'
						manipulation = To Manipulation... 0.01 75 600
						plus DurationTier 'name$'
						Replace duration tier
						select PitchTier fittedf0_st
						Copy... fittedf0_hz
						Formula... exp(self*ln(2)/12)
						plus manipulation
						Replace pitch tier
						select manipulation
						resyn = Get resynthesis (overlap-add)
					else
						select Sound 'name$'
						manipulation = To Manipulation... 0.01 75 600
						select PitchTier fittedf0_st
						Copy... fittedf0_hz
						Formula... exp(self*ln(2)/12)
						plus manipulation
						Replace pitch tier
						select manipulation
						resyn = Get resynthesis (overlap-add)
						Extract part... t1 t2 Rectangular 1.0 yes
					endif
					Play
					plus resyn
					nocheck plus manipulation
					Remove
					demo Paint rounded rectangle... {0.98,0.98,0.8} 64 74 42 45 2
					demo Black
					demo Draw rounded rectangle... 64 74 42 45 2
					demo Text special... 69 centre 43.5 half Times 10 0 Play resynthesis
				else
					demo Paint rectangle... White 0 100 25 30
					demo Red
					demo Text special... 2 left 27 bottom Courier 14 0 No resynthesis...
				endif
			elsif demoClickedIn (76, 82, 42, 45)		; clicked Previous
				if current_file < 2
					more_file -= 2
					demo Paint rectangle... White 0 100 25 30
					demo Red
					demo Text special... 2 left 27 bottom Courier 14 0 First sound file...
					if found_interval and more_file >= -1
						select PitchTier semitonef0
						plus PitchTier fittedf0_st
						plus Manipulation 'name$'
					Remove
					endif
				else
					current_file -= 2
					demo Erase all
					loop = 0
					replot = 0
					replotting = 0
					demo Text special... 30 left 27 bottom Courier 14 0 Synthesizing previes sound...
					call Save 'directory$' 'name$'
					if found_interval
						select TextGrid 'name$'
						plus PitchTier 'name$'
						plus PitchTier semitonef0
						plus PitchTier fittedf0_st
						plus Manipulation 'name$'
						plus Pitch semitonef0
						nocheck plus Pitch fittedf0_st
					else
						select PitchTier 'name$'
						plus PitchTier samplef0
						plus TableOfReal samplef0
					endif
					plus PointProcess 'name$'
					Remove
					more_file = 1
				endif
			elsif demoClickedIn (83, 87, 42, 45)		; clicked Next
				clicked_next = 1
				if current_file = numberOfFiles
					more_file = 0
					call Save 'directory$' 'name$'
				else
					demo Erase all
					loop = 0
					replot = 0
					repolotting = 0
					demo Text special... 30 left 27 bottom Courier 14 0 Synthesizing next sound...
					call Save 'directory$' 'name$'
					select TextGrid 'name$'
					plus PitchTier 'name$'
					plus Manipulation 'name$'
					plus PitchTier semitonef0
					plus PitchTier fittedf0_st
;					plus Pitch fittedf0_st
					plus PointProcess 'name$'
					plus Pitch semitonef0
					more_file = 1
				endif
				if demoClickedIn (83, 87, 42, 45)
# To prevent clicking Previous from leading here mysteriously
					Remove
				endif
			elsif demoClickedIn (89, 93, 42, 45)		; clicked Exit
				demo Paint rectangle... White 0 100 25 30
				demo Red
				demo Text special... 30 left 27 bottom Courier 14 0 Now close window manually...
				loop = 0
				select TextGrid 'name$'
				plus PointProcess 'name$'
				plus Strings list
				plus Sound 'name$'
				plus Pitch semitonef0
				if found_interval and more_file >= 0
					plus PitchTier 'name$'
					plus Manipulation 'name$'
					plus PitchTier semitonef0
					plus PitchTier fittedf0_st
;					plus Pitch fittedf0_st
					plus Pitch semitonef0
					plus Manipulation 'name$'
					plus PitchTier fittedf0_st
					plus Manipulation 'name$'
				elsif  more_file >= 0
					plus PitchTier 'name$'
					plus TextGrid 'name$'
					plus PitchTier samplef0
					plus TableOfReal samplef0
					plus PitchTier semitonef0
				endif
				plus PitchTier 'name$'
				Remove
				call RemoveObject PitchTier semitonef0
				call RemoveObject Pitch simitonef0
				call RemoveObject PitchTier fittedf0_st
				call RemoveObject Pitch fittedf0_st
				call RemoveObject Manipulation 'name$'
				exit
			elsif demoClickedIn (5, 10, 42, 45)		; clicked Replot
				replot = 1
				demo Text special... 30 left 27 bottom Courier 14 0 Re-synthesizing...
				replotting = 1
				call Save 'directory$' 'name$'
				if found_interval
					select Pitch semitonef0
;					plus Pitch fittedf0_st
				else
					select PitchTier 'name$'
					plus PitchTier samplef0
					plus TableOfReal samplef0
				endif
				Remove
				replotting = 0
			endif
		endif
		demo Select inner viewport... 0 100 0 100
		demo Axes... 0 100 0 100
	endwhile
endproc

procedure qTA_synthesis
	select TextGrid 'name$'
	nintervals = Get number of intervals... 1
	sampleStart = Get starting point... 1 1
	sampleEnd = Get end point... 1 nintervals
	Create PitchTier... fittedf0_st sampleStart sampleEnd
	Create PitchTier... fittedvelocity sampleStart sampleEnd
	Create TableOfReal... fittedf0_st 1 2
	Set column label (index)... 1 SampleTime
	Set column label (index)... 2 fittedf0 (st)
	Create TableOfReal... fittedvelocity 1 2
	Set column label (index)... 1 SampleTime
	Set column label (index)... 2 velocity (st/s)
	
	dur_change_total = 0
	
	nrows = 0
	interval = 0
	total_samples = 0
	select TextGrid 'name$'
	label$ = Get label of interval... 1 1
	if label$ = ""
		xInitial$ = Get label of interval... 4 1
		xInitial = number(xInitial$)
	else
		start = Get starting point... 1 1
		select PitchTier semitonef0
		index_first = Get low index from time... start
		xInitial = Get value at index... index_first
	endif
	for n from 1 to nintervals
		select TextGrid 'name$'
		label$ = Get label of interval... 1 n
		xInitial$ = Get label of interval... 4 1
		xInitial = number(xInitial$)
		xInitialNew = xInitial
		if not label$ = "" and label$ != "'Silence_marker$'" and interval_dur >= 0.001
			interval = interval + 1
			start = Get starting point... 1 n
			end = Get end point... 1 n
			if interval == 1
				tInitial = start
			endif
			original_dur'interval' = end - start
			select TextGrid 'name$'
			m$ = Get label of interval... 3 n
			b$ = Get label of interval... 4 n
			lambda$ = Get label of interval... 5 n
			m = number(m$)
			b = number(b$)
			lambda = number(lambda$)
			duration$ = Get label of interval... 6 n
			duration = number(duration$)
			duration *= 0.001
			dur_ratio'interval' = duration / original_dur'interval'
			dur_change'interval' = duration - original_dur'interval'
			if dur_change'interval' < 0.001 and dur_change'interval' > -0.001
				dur_change'interval' = 0
			endif
			mNew'interval' = m
			bNew'interval' = b
			lambdaNew'interval' = lambda
			durationNew'interval' = duration

			select PitchTier 'name$'
			start += dur_change_total
			if interval == 1
				x0 = xInitial
				v0 = 0
				a0 = 0
			else
				x0 = xFinal
				v0 = vFinal
				a0 = aFinal
			endif

			select PitchTier fittedf0_st
			Remove points between... start end
			select PitchTier fittedvelocity
			nsamples = duration * f0_sample_rate
			call qTA m b-m*duration+xInitial lambda nsamples
			
			select PitchTier 'name$'
			index_first = Get high index from time... start
			index_last = Get low index from time... end
			ssef0 = 0
			for x from index_first to index_last
				select PitchTier 'name$'
				sampletime = Get time from index... x
				rawf0 = Get value at time... sampletime
				if rawf0 = undefined
					index_last = index_last - 1
				endif
				select PitchTier fittedf0_st
				qTAf0 = Get value at time... sampletime
				ssef0 += (qTAf0 - rawf0)^2
			endfor
			rmse = sqrt(ssef0 / (index_last - index_first))

			select PitchTier fittedf0_st
			first_index = Get nearest index from time... start
			for x from first_index to first_index + nsamples
				select PitchTier fittedf0_st
				qTAf0 = Get value at index... x
				if not qTAf0 = undefined
					sampletime = Get time from index... x
					select TableOfReal fittedf0_st
					nrows = Get number of rows
					Set value... nrows 1 sampletime
					Set value... nrows 2 qTAf0
					Set row label (index)... nrows 'label$'
					Insert row (index)... nrows + 1
					select PitchTier fittedvelocity
					qTAvelocity = Get value at index... x
					select TableOfReal fittedvelocity
					nrows = Get number of rows
					Set value... nrows 1 sampletime
					Set value... nrows 2 qTAvelocity
					Set row label (index)... nrows 'label$'
					Insert row (index)... nrows + 1
				endif
				total_samples += total_samples
			endfor
			tFinal = end
			dur_change_total += dur_change'interval'
		endif
	endfor
		
	select TableOfReal fittedf0_st
	if nrows > total_samples
		Remove row (index)... nrows + 1
	endif
	select TableOfReal fittedvelocity
	if nrows > total_samples
		Remove row (index)... nrows + 1
	endif
	if dur_change_total > 0
		tFinal += dur_change_total
	endif
endproc

procedure qTA mn bb lambda nsamples
	delta = 1/f0_sample_rate
	t = 0
	c1 = x0 - bb
	c2 = v0 + c1 * lambda - mn
	c3 = (a0 + 2 * c2 * lambda - c1 * lambda^2) / 2

	for i from 1 to nsamples
		t += delta
		y = (mn * t + bb) + (c1 + c2 * t  + c3 * t^2) * exp(-lambda * t)
		v = -lambda * (c3 * t^2 + c2 * t + c1) * exp(-lambda * t) + (2 * c3 * t + c2) * exp(-lambda * t) + mn
		select PitchTier fittedf0_st
		Add point... t+start y
		select PitchTier fittedvelocity
		Add point... t+start v		
	endfor
	xFinal = y
	vFinal = v
	aFinal = lambda^2 * (c3 * t^2 + c2 * t + c1) * exp(-lambda * t) - 2 * lambda * (2 * c3 * t + c2) * exp(-lambda * t) + 2 * c3 * exp(-lambda * t)
endproc

procedure RemoveObject objType$ objName$
	select all
	nobjects = numberOfSelected()
	for obj from 1 to nobjects
		obj_nanme$ = selected$(obj)
		if obj_nanme$ = "'objType$' 'objName$'"
			select 'objType$' 'objName$'
			Remove
			nobjects -= 1
			select all
		endif
	endfor
	selectObject ()
endproc